/**
 * OAuth Connect Component
 */
import { useState, useEffect } from 'react';

const OAuthConnect = ({ onConnectionChange }) => {
  const [isConnected, setIsConnected] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [connectionStatus, setConnectionStatus] = useState(null);
  const [error, setError] = useState(null);

  useEffect(() => {
    checkConnectionStatus();
  }, []);

  const checkConnectionStatus = async () => {
    setIsLoading(true);
    setError(null);

    try {
      const response = await fetch(
        `${window.samurAiAdmin.apiUrl}/oauth/status`,
        {
          headers: {
            'X-WP-Nonce': window.samurAiAdmin.nonce,
          },
        }
      );

      const data = await response.json();

      if (data.connected) {
        setIsConnected(true);
        setConnectionStatus(data);
      }

      // Notify parent of connection change
      if (onConnectionChange) {
        onConnectionChange();
      }
    } catch (err) {
      console.error('Error checking OAuth status:', err);
    } finally {
      setIsLoading(false);
    }
  };

  const handleConnect = () => {
    setError(null);

    // Build OAuth URL using localized base URL
    const callbackUrl = `${window.location.origin}/samuraiauth/?nonce=${window.samurAiAdmin.nonce}`;
    const oauthBaseUrl = window.samurAiAdmin.oauthBaseUrl || 'https://cp.websamurai.io';
    const oauthUrl = `${oauthBaseUrl}/oauth/auth/?callback=${encodeURIComponent(callbackUrl)}`;

    // Redirect to OAuth provider
    window.location.href = oauthUrl;
  };

  const handleRefresh = async () => {
    setIsLoading(true);
    setError(null);

    try {
      const response = await fetch(
        `${window.samurAiAdmin.apiUrl}/oauth/refresh`,
        {
          method: 'POST',
          headers: {
            'X-WP-Nonce': window.samurAiAdmin.nonce,
            'Content-Type': 'application/json',
          },
        }
      );

      const data = await response.json();

      if (data.success) {
        // Refresh the connection status to get updated user data
        await checkConnectionStatus();
      } else {
        setError(data.message || 'Failed to refresh user data');
      }
    } catch (err) {
      setError('An error occurred while refreshing user data');
      console.error('Error refreshing:', err);
    } finally {
      setIsLoading(false);
    }
  };

  const handleDisconnect = async () => {
    if (!confirm('Are you sure you want to disconnect from cp.websamurai.io?')) {
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const response = await fetch(
        `${window.samurAiAdmin.apiUrl}/oauth/disconnect`,
        {
          method: 'POST',
          headers: {
            'X-WP-Nonce': window.samurAiAdmin.nonce,
            'Content-Type': 'application/json',
          },
        }
      );

      const data = await response.json();

      if (data.success) {
        setIsConnected(false);
        setConnectionStatus(null);

        // Notify parent of connection change
        if (onConnectionChange) {
          onConnectionChange();
        }
      } else {
        setError(data.message || 'Failed to disconnect');
      }
    } catch (err) {
      setError('An error occurred while disconnecting');
      console.error('Error disconnecting:', err);
    } finally {
      setIsLoading(false);
    }
  };

  if (isLoading) {
    return (
      <div className="oauth-connect">
        <div className="loading">Loading connection status...</div>
      </div>
    );
  }

  return (
    <div className="oauth-connect">
      <div className="oauth-header">
        <h3>Connect to Service</h3>
        <p>
          Authenticate with your SamuraiWP Control Panel account to enable
          advanced AI features and synchronization.
        </p>
      </div>

      {error && (
        <div className="oauth-error">
          <strong>Error:</strong> {error}
        </div>
      )}

      {isConnected ? (
        <div className="oauth-connected">
          <div className="status-badge connected">
            <span className="status-icon">✓</span>
            Connected
          </div>

          <div className="connection-info">
            <div className="connection-info-header">
              <h4>Connection Details</h4>
              <button
                className="button button-small oauth-refresh"
                onClick={handleRefresh}
                disabled={isLoading}
                title="Refresh user data from server"
              >
                ↻ Refresh
              </button>
            </div>
            {connectionStatus && (
              <dl>
                <dt>Connected:</dt>
                <dd>
                  {connectionStatus.connected_at
                    ? new Date(connectionStatus.connected_at).toLocaleString()
                    : 'N/A'}
                </dd>
                {connectionStatus.user_data && (
                  <>
                    <dt>User ID:</dt>
                    <dd>{connectionStatus.user_data.id || 'N/A'}</dd>
                    <dt>Name:</dt>
                    <dd>
                      {connectionStatus.user_data.FirstName}{' '}
                      {connectionStatus.user_data.LastName}
                    </dd>
                    <dt>Login ID:</dt>
                    <dd>{connectionStatus.user_data.LoginID || 'N/A'}</dd>
                    <dt>User Email:</dt>
                    <dd>{connectionStatus.user_data.Email || 'N/A'}</dd>
                    <dt>Role:</dt>
                    <dd>{connectionStatus.user_data.RoleList || 'N/A'}</dd>
                    <dt>Last Refreshed:</dt>
                    <dd>
                      {connectionStatus.refreshed_at
                        ? new Date(connectionStatus.refreshed_at).toLocaleString()
                        : 'Never'}
                    </dd>
                  </>
                )}
              </dl>
            )}
          </div>

          <button
            className="button button-secondary oauth-disconnect"
            onClick={handleDisconnect}
            disabled={isLoading}
          >
            Disconnect
          </button>
        </div>
      ) : (
        <div className="oauth-disconnected">
          <div className="status-badge disconnected">
            <span className="status-icon">○</span>
            Not Connected
          </div>

          <div className="oauth-benefits">
            <h4>Benefits of Connecting:</h4>
            <ul>
              <li>Access advanced AI models and features</li>
              <li>Sync your settings across multiple sites</li>
              <li>Automatic updates and improvements</li>
              <li>Priority support and assistance</li>
            </ul>
          </div>

          <button
            className="button button-primary oauth-connect-btn"
            onClick={handleConnect}
            disabled={isLoading}
          >
            Connect to Service
          </button>

          <p className="oauth-privacy">
            Your authentication is secure and encrypted. We only access the
            information necessary to provide our services.
          </p>
        </div>
      )}
    </div>
  );
};

export default OAuthConnect;
