<?php
/**
 * SamuraiSite GraphQL Operations
 *
 * Handles SamuraiSite record management via GraphQL
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Make a GraphQL request to the SamuraiWP Control Panel
 *
 * @param string $query The GraphQL query or mutation.
 * @param array  $variables The variables for the query.
 * @return array|WP_Error Response data or WP_Error on failure.
 */
function websamurai_graphql_request( $query, $variables = array() ) {
	$auth_key = get_option( 'websamurai_cp_auth_key' );

	if ( empty( $auth_key ) ) {
		return new WP_Error( 'not_connected', 'Not connected to SamuraiWP Control Panel' );
	}

	// Determine the GraphQL endpoint URL based on LOCALDEV constant.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	// Build the GraphQL request body.
	$graphql_body = array(
		'query' => $query,
	);

	if ( ! empty( $variables ) ) {
		$graphql_body['variables'] = $variables;
	}

	// Make the request to the GraphQL endpoint.
	$response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $graphql_body ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( 200 !== $status_code ) {
		return new WP_Error( 'http_error', 'Server returned status: ' . $status_code, $data );
	}

	// Check for GraphQL errors.
	if ( isset( $data['errors'] ) && ! empty( $data['errors'] ) ) {
		$err = array_map( function( $error ) {
			return isset( $error['message'] ) ? $error['message'] : 'Unknown error';
		}, $data['errors'] );
		return new WP_Error( 'graphql_error', 'Error:' . implode( '\n', $err ) );
	}

	return $data;
}

/**
 * Get SamuraiSite record by ID
 *
 * @param int $site_id The SamuraiSite ID.
 * @return array|WP_Error SamuraiSite data or WP_Error on failure.
 */
function websamurai_get_samurai_site( $site_id ) {
	$query = 'query ($id: ID!) {
		SamuraiSite: SamuraiSite(id: $id) {
			id
			DomainName
			Data
			User
			Deleted
			PromptInstructions
			PromptInstructionsImage
		}
	}';

	$variables = array(
		'id' => $site_id,
	);

	$response = websamurai_graphql_request( $query, $variables );

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	// Extract SamuraiSite data from the response.
	$site_data = isset( $response['data']['SamuraiSite'] ) ? $response['data']['SamuraiSite'] : null;

	if ( ! $site_data ) {
		return new WP_Error( 'no_data', 'No SamuraiSite data received from server' );
	}
	if( $site_data['Deleted'] ) {
		return new WP_Error( 'not_found', 'SamuraiSite not found (deleted)' );
	}

	return $site_data;
}

/**
 * Create a new SamuraiSite record
 *
 * @param array $site_data Site data (DomainName, Data, MCPAPIToken).
 * @return array|WP_Error Created SamuraiSite data or WP_Error on failure.
 */
function websamurai_create_samurai_site( $site_data ) {
	$mutation = 'mutation ($id: ID, $DomainName: String, $Data: CustomObject, $User: ID, $Deleted: Boolean, $ModelLinks: AnyObjectType, $MCPAPIToken: String) {
		CreateSamuraiSite (id: $id, DomainName: $DomainName, Data: $Data, User: $User, Deleted: $Deleted, ModelLinks: $ModelLinks, MCPAPIToken: $MCPAPIToken) {
			id
			DomainName
			Data
			User
		}
	}';

	$variables = array(
		'DomainName' => isset( $site_data['DomainName'] ) ? $site_data['DomainName'] : '',
		'Data'       => isset( $site_data['Data'] ) ? $site_data['Data'] : null,
	);

	// Add MCPAPIToken if provided.
	if ( isset( $site_data['MCPAPIToken'] ) ) {
		$variables['MCPAPIToken'] = $site_data['MCPAPIToken'];
	}

	$response = websamurai_graphql_request( $mutation, $variables );

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	// Extract created SamuraiSite data from the response.
	$created_site = isset( $response['data']['CreateSamuraiSite'] ) ? $response['data']['CreateSamuraiSite'] : null;

	if ( ! $created_site ) {
		return new WP_Error( 'create_failed', 'Failed to create SamuraiSite' );
	}

	return $created_site;
}

/**
 * Update an existing SamuraiSite record
 *
 * @param int   $site_id The SamuraiSite ID to update.
 * @param array $site_data Site data to update (DomainName, Data, MCPAPIToken, PromptInstructions, PromptInstructionsImage).
 * @return array|WP_Error Updated SamuraiSite data or WP_Error on failure.
 */
function websamurai_update_samurai_site( $site_id, $site_data ) {
	$mutation = 'mutation ($id: ID!, $DomainName: String, $Data: CustomObject, $User: ID, $Deleted: Boolean, $ModelLinks: AnyObjectType, $MCPAPIToken: String, $PromptInstructions: String, $PromptInstructionsImage: String) {
		UpdateSamuraiSite (id: $id, DomainName: $DomainName, Data: $Data, User: $User, Deleted: $Deleted, ModelLinks: $ModelLinks, MCPAPIToken: $MCPAPIToken, PromptInstructions: $PromptInstructions, PromptInstructionsImage: $PromptInstructionsImage) {
			id
			DomainName
			Data
			User
			PromptInstructions
			PromptInstructionsImage
		}
	}';

	$variables = array(
		'id' => $site_id,
	);
	if ( isset( $site_data['DomainName'] ) ) {
		$variables['DomainName'] = $site_data['DomainName'];
	}
	if ( isset( $site_data['Data'] ) ) {
		$variables['Data'] = $site_data['Data'];
	}

	// Only add MCPAPIToken if provided.
	if ( isset( $site_data['MCPAPIToken'] ) ) {
		$variables['MCPAPIToken'] = $site_data['MCPAPIToken'];
	}

	// Only add PromptInstructions if provided.
	if ( isset( $site_data['PromptInstructions'] ) ) {
		$variables['PromptInstructions'] = $site_data['PromptInstructions'];
	}

	// Only add PromptInstructionsImage if provided.
	if ( isset( $site_data['PromptInstructionsImage'] ) ) {
		$variables['PromptInstructionsImage'] = $site_data['PromptInstructionsImage'];
	}

	$response = websamurai_graphql_request( $mutation, $variables );

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	// Extract updated SamuraiSite data from the response.
	$updated_site = isset( $response['data']['UpdateSamuraiSite'] ) ? $response['data']['UpdateSamuraiSite'] : null;

	if ( ! $updated_site ) {
		return new WP_Error( 'update_failed', 'Failed to update SamuraiSite' );
	}

	return $updated_site;
}

/**
 * Sync SamuraiSite record - get existing or create new
 *
 * This should be called during user refresh to ensure the site is registered.
 *
 * @return array|WP_Error|false SamuraiSite data, WP_Error on failure, or false if disconnected.
 */
function websamurai_sync_samurai_site() {
	$auth_key = get_option( 'websamurai_cp_auth_key' );

	// If not connected, return false.
	if ( empty( $auth_key ) ) {
		return false;
	}

	$saved_site_id = get_option( 'websamurai_cp_site_id' );

	// Prepare current site data.
	$site_name = get_bloginfo( 'name' );
	$site_url  = get_site_url();
	$domain    = wp_parse_url( $site_url, PHP_URL_HOST );

	// Get or create the MCP API key for remote access.
	$mcp_api_token = WebSamurai_API_Keys::get_or_create_samurai_remote_key();

	$current_site_data = array(
		'DomainName'  => $domain,
		'Data'        => array(
			'SiteName' => $site_name,
			'SiteURL'  => $site_url,
		),
		'MCPAPIToken' => $mcp_api_token,
	);

	// If we have a saved site ID, try to fetch and update it.
	if ( ! empty( $saved_site_id ) ) {
		$site_data = websamurai_get_samurai_site( $saved_site_id );

		// If we got valid site data, update it with current information.
		if ( ! is_wp_error( $site_data ) ) {
			$updated_site = websamurai_update_samurai_site( $saved_site_id, $current_site_data );

			if ( is_wp_error( $updated_site ) ) {
				// Continue with existing data even if update failed.
				update_option( 'websamurai_cp_site_synced_at', current_time( 'mysql' ) );
				return $site_data;
			}

			update_option( 'websamurai_cp_site_synced_at', current_time( 'mysql' ) );
			return $updated_site;
		}

		// If the site doesn't exist (404 or no data), we'll create a new one below.
	}

	// Create a new SamuraiSite using the prepared data.
	$created_site = websamurai_create_samurai_site( $current_site_data );

	if ( is_wp_error( $created_site ) ) {

		// Disconnect on failure to create.
		websamurai_disconnect_oauth();

		return $created_site;
	}

	// Save the new site ID.
	update_option( 'websamurai_cp_site_id', $created_site['id'] );
	update_option( 'websamurai_cp_site_synced_at', current_time( 'mysql' ) );

	return $created_site;
}

/**
 * Disconnect OAuth and clear all related data
 *
 * Note: We preserve the site ID so that reconnecting uses the same SamuraiSite record.
 *
 * @return void
 */
function websamurai_disconnect_oauth() {
	delete_option( 'websamurai_cp_auth_key' );
	delete_option( 'websamurai_cp_connected_at' );
	delete_option( 'websamurai_cp_user_data' );
	delete_option( 'websamurai_cp_user_refreshed_at' );
	delete_option( 'websamurai_cp_site_synced_at' );

	// Note: We intentionally keep 'websamurai_cp_site_id' so reconnecting reuses the same site.
}
