<?php
/**
 * OAuth Handler
 *
 * Handles OAuth callbacks and authentication with cp.websamurai.io
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register custom rewrite rule for OAuth callback
 *
 * @return void
 */
function websamurai_oauth_rewrite_rules() {
	add_rewrite_rule( '^samuraiauth/?', 'index.php?websamurai_oauth_callback=1', 'top' );
}
add_action( 'init', 'websamurai_oauth_rewrite_rules' );

/**
 * Register custom query var
 *
 * @param array $vars Query vars.
 * @return array
 */
function websamurai_oauth_query_vars( $vars ) {
	$vars[] = 'websamurai_oauth_callback';
	return $vars;
}
add_filter( 'query_vars', 'websamurai_oauth_query_vars' );

/**
 * Handle OAuth callback
 *
 * @return void
 */
function websamurai_handle_oauth_callback() {
	$callback = get_query_var( 'websamurai_oauth_callback' );

	if ( $callback ) {
		if ( current_user_can( 'manage_options' ) ) {
			$nonce = isset( $_GET['nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['nonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
				wp_die( 'OAuth Error: Invalid nonce. Please try again.' );
			}

			// Get the key parameter from the URL.
			$key = isset( $_GET['key'] ) ? sanitize_text_field( wp_unslash( $_GET['key'] ) ) : '';

			if ( empty( $key ) ) {
				wp_die( 'OAuth Error: No authentication key provided.' );
			}

			// Store the auth key in options table.
			update_option( 'websamurai_cp_auth_key', $key );

			// Store connection timestamp.
			update_option( 'websamurai_cp_connected_at', current_time( 'mysql' ) );

			// Refresh user data from OAuth server.
			$user_data = websamurai_perform_user_refresh();

			// Sync SamuraiSite record.
			$site_result = websamurai_sync_samurai_site();

			if ( is_wp_error( $site_result ) ) {
				// If sync failed and we disconnected, redirect to show error.
				if ( empty( get_option( 'websamurai_cp_auth_key' ) ) ) {
					wp_die( 'OAuth Error: Failed to register site with SamuraiWP Control Panel. Please try again.<br>' . $site_result->get_error_message() );
				}
			}

			// Redirect to admin page with success message.
			$redirect_url = add_query_arg(
				array(
					'page'          => 'websamurai',
					'oauth_success' => '1',
				),
				admin_url( 'admin.php' )
			);

			wp_safe_redirect( $redirect_url );
			exit;
		}
	}
}
add_action( 'template_redirect', 'websamurai_handle_oauth_callback' );

/**
 * Show admin notice after successful OAuth connection
 *
 * @return void
 */
function websamurai_oauth_admin_notices() {
	if ( isset( $_GET['oauth_success'] ) && '1' === $_GET['oauth_success'] ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		?>
		<div class="notice notice-success is-dismissible">
			<p><strong>Success!</strong> You have successfully connected to SamuraiWP Control Panel.</p>
		</div>
		<?php
	}
}
add_action( 'admin_notices', 'websamurai_oauth_admin_notices' );

/**
 * Flush rewrite rules on plugin activation
 *
 * @return void
 */
function websamurai_oauth_activate() {
	websamurai_oauth_rewrite_rules();
	flush_rewrite_rules();
}
register_activation_hook( WEBSAMURAI_PLUGIN_FILE, 'websamurai_oauth_activate' );

/**
 * Flush rewrite rules on plugin deactivation
 *
 * @return void
 */
function websamurai_oauth_deactivate() {
	flush_rewrite_rules();
}
register_deactivation_hook( WEBSAMURAI_PLUGIN_FILE, 'websamurai_oauth_deactivate' );
