<?php
/**
 * OAuth REST API Endpoints
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register OAuth REST API endpoints
 *
 * @return void
 */
function websamurai_register_oauth_routes() {
	register_rest_route(
		'websamurai/v1',
		'/oauth/status',
		array(
			'methods'             => 'GET',
			'callback'            => 'websamurai_oauth_status',
			'permission_callback' => function () {
				return current_user_can( 'manage_options' );
			},
		)
	);

	register_rest_route(
		'websamurai/v1',
		'/oauth/disconnect',
		array(
			'methods'             => 'POST',
			'callback'            => 'websamurai_oauth_disconnect',
			'permission_callback' => function () {
				return current_user_can( 'manage_options' );
			},
		)
	);

	register_rest_route(
		'websamurai/v1',
		'/oauth/refresh',
		array(
			'methods'             => 'POST',
			'callback'            => 'websamurai_oauth_refresh_user',
			'permission_callback' => function () {
				return current_user_can( 'manage_options' );
			},
		)
	);

	register_rest_route(
		'websamurai/v1',
		'/prompt-instructions',
		array(
			'methods'             => 'GET',
			'callback'            => 'websamurai_get_prompt_instructions',
			'permission_callback' => function () {
				return current_user_can( 'manage_options' );
			},
		)
	);

	register_rest_route(
		'websamurai/v1',
		'/prompt-instructions',
		array(
			'methods'             => 'POST',
			'callback'            => 'websamurai_save_prompt_instructions',
			'permission_callback' => function () {
				return current_user_can( 'manage_options' );
			},
		)
	);
}
add_action( 'rest_api_init', 'websamurai_register_oauth_routes' );

/**
 * Get OAuth connection status
 *
 * @return WP_REST_Response
 */
function websamurai_oauth_status() {
	$auth_key     = get_option( 'websamurai_cp_auth_key' );
	$connected_at = get_option( 'websamurai_cp_connected_at' );

	if ( ! empty( $auth_key ) ) {
		$user_data    = get_option( 'websamurai_cp_user_data' );
		$refreshed_at = get_option( 'websamurai_cp_user_refreshed_at' );

		return new WP_REST_Response(
			array(
				'connected'    => true,
				'connected_at' => $connected_at,
				'email'        => get_option( 'admin_email' ),
				'account_id'   => substr( $auth_key, 0, 8 ) . '...',
				'user_data'    => $user_data,
				'refreshed_at' => $refreshed_at,
			),
			200
		);
	}

	return new WP_REST_Response(
		array(
			'connected' => false,
		),
		200
	);
}

/**
 * Disconnect OAuth connection
 *
 * @return WP_REST_Response
 */
function websamurai_oauth_disconnect() {
	// Use centralized disconnect function.
	websamurai_disconnect_oauth();

	return new WP_REST_Response(
		array(
			'success' => true,
			'message' => 'Successfully disconnected from SamuraiWP Control Panel',
		),
		200
	);
}

/**
 * Perform user data refresh (internal helper)
 *
 * @return array|WP_Error User data on success, WP_Error on failure.
 */
function websamurai_perform_user_refresh() {
	$auth_key = get_option( 'websamurai_cp_auth_key' );

	if ( empty( $auth_key ) ) {
		return new WP_Error( 'not_connected', 'Not connected to SamuraiWP Control Panel' );
	}

	// Determine the GraphQL endpoint URL based on LOCALDEV constant.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	// Build the GraphQL query.
	$graphql_query = array(
		'query'     => 'query ($id: ID) {
			User: User(id: $id) {
				id
				FirstName
				LastName
				LoginID
				Email
				Password
				Special
				RoleList
				AddedDate
				UpdatedDate
				Deleted
			}
		}',
		'variables' => array(
			'id' => -1,
		),
	);

	// Make the request to the GraphQL endpoint.
	$response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $graphql_query ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( 200 !== $status_code ) {
		return new WP_Error( 'http_error', 'Server returned status: ' . $status_code, $data );
	}

	// Check for GraphQL errors.
	if ( isset( $data['errors'] ) && ! empty( $data['errors'] ) ) {
		return new WP_Error( 'graphql_error', 'GraphQL query error', $data['errors'] );
	}

	// Extract user data from the response.
	$user_data = isset( $data['data']['User'] ) ? $data['data']['User'] : null;

	if ( ! $user_data ) {
		return new WP_Error( 'no_user_data', 'No user data received from server' );
	}

	// Store user data in options for future reference.
	update_option( 'websamurai_cp_user_data', $user_data );
	update_option( 'websamurai_cp_user_refreshed_at', current_time( 'mysql' ) );

	return $user_data;
}

/**
 * Refresh user data from OAuth server (REST API endpoint)
 *
 * @return WP_REST_Response
 */
function websamurai_oauth_refresh_user() {
	$user_data = websamurai_perform_user_refresh();

	if ( is_wp_error( $user_data ) ) {
		$error_data  = $user_data->get_error_data();
		$status_code = is_array( $error_data ) ? 500 : 400;

		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => $user_data->get_error_message(),
				'errors'  => $error_data,
			),
			$status_code
		);
	}

	return new WP_REST_Response(
		array(
			'success' => true,
			'message' => 'User data refreshed successfully',
			'data'    => $user_data,
		),
		200
	);
}

/**
 * Get prompt instructions from SamuraiSite
 *
 * @return WP_REST_Response
 */
function websamurai_get_prompt_instructions() {
	$auth_key = get_option( 'websamurai_cp_auth_key' );

	if ( empty( $auth_key ) ) {
		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => 'Not connected to SamuraiWP Control Panel',
			),
			400
		);
	}

	$site_id = get_option( 'websamurai_cp_site_id' );

	if ( empty( $site_id ) ) {
		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => 'No site ID found',
			),
			400
		);
	}

	// Get the SamuraiSite data.
	$site_data = websamurai_get_samurai_site( $site_id );

	if ( is_wp_error( $site_data ) ) {
		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => $site_data->get_error_message(),
			),
			500
		);
	}

	return new WP_REST_Response(
		array(
			'success' => true,
			'data'    => array(
				'PromptInstructions'      => isset( $site_data['PromptInstructions'] ) ? $site_data['PromptInstructions'] : '',
				'PromptInstructionsImage' => isset( $site_data['PromptInstructionsImage'] ) ? $site_data['PromptInstructionsImage'] : '',
			),
		),
		200
	);
}

/**
 * Save prompt instructions to SamuraiSite
 *
 * @param WP_REST_Request $request The request object.
 * @return WP_REST_Response
 */
function websamurai_save_prompt_instructions( $request ) {
	$auth_key = get_option( 'websamurai_cp_auth_key' );

	if ( empty( $auth_key ) ) {
		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => 'Not connected to SamuraiWP Control Panel',
			),
			400
		);
	}

	$site_id = get_option( 'websamurai_cp_site_id' );

	if ( empty( $site_id ) ) {
		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => 'No site ID found',
			),
			400
		);
	}

	// Get the prompt instructions from the request.
	$prompt_instructions       = $request->get_param( 'promptInstructions' );
	$prompt_instructions_image = $request->get_param( 'promptInstructionsImage' );

	// Update the SamuraiSite with the new prompt instructions.
	$update_data = array(
		'PromptInstructions'      => null !== $prompt_instructions ? $prompt_instructions : '',
		'PromptInstructionsImage' => null !== $prompt_instructions_image ? $prompt_instructions_image : '',
	);

	$updated_site = websamurai_update_samurai_site( $site_id, $update_data );

	if ( is_wp_error( $updated_site ) ) {
		return new WP_REST_Response(
			array(
				'success' => false,
				'message' => $updated_site->get_error_message(),
			),
			500
		);
	}

	return new WP_REST_Response(
		array(
			'success' => true,
			'message' => 'Prompt instructions saved successfully',
			'data'    => array(
				'PromptInstructions'      => isset( $updated_site['PromptInstructions'] ) ? $updated_site['PromptInstructions'] : '',
				'PromptInstructionsImage' => isset( $updated_site['PromptInstructionsImage'] ) ? $updated_site['PromptInstructionsImage'] : '',
			),
		),
		200
	);
}
