<?php
/**
 * MCP Tool: Theme Information
 *
 * Reads information about the active WordPress theme.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Theme Info Tool Class
 */
class WebSamurai_MCP_Theme_Info_Tool {

	/**
	 * Get tool definition
	 *
	 * @return array
	 */
	public static function get_definition() {
		return array(
			'name'        => 'theme_info',
			'description' => 'Read information about the active WordPress theme including name, version, description, author, and other details',
			'inputSchema' => array(
				'type'       => 'object',
				'properties' => array(
					'stylesheet' => array(
						'type'        => 'string',
						'description' => 'The stylesheet (theme) slug. Defaults to current active theme.',
					),
				),
				'required'   => array(),
			),
		);
	}

	/**
	 * Execute the tool
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	public static function execute( $arguments ) {
		return self::get_theme_info( $arguments );
	}

	/**
	 * Get theme information
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function get_theme_info( $arguments ) {
		$stylesheet = isset( $arguments['stylesheet'] ) ? sanitize_text_field( $arguments['stylesheet'] ) : get_stylesheet();

		// Get theme object.
		$theme = wp_get_theme( $stylesheet );

		// Check if theme exists.
		if ( ! $theme->exists() ) {
			return self::error_response( sprintf( 'Theme not found: %s', $stylesheet ) );
		}

		// Gather theme information.
		$theme_info = array(
			'Name'        => $theme->get( 'Name' ),
			'Version'     => $theme->get( 'Version' ),
			'Description' => $theme->get( 'Description' ),
			'Author'      => $theme->get( 'Author' ),
			'AuthorURI'   => $theme->get( 'AuthorURI' ),
			'ThemeURI'    => $theme->get( 'ThemeURI' ),
			'Template'    => $theme->get( 'Template' ),
			'Status'      => $theme->get( 'Status' ),
			'Tags'        => $theme->get( 'Tags' ),
			'TextDomain'  => $theme->get( 'TextDomain' ),
			'DomainPath'  => $theme->get( 'DomainPath' ),
		);

		// Check if it's a child theme.
		$is_child_theme = ! empty( $theme_info['Template'] );

		// Build output message.
		$output  = sprintf( "Theme Information: %s\n\n", $theme_info['Name'] );
		$output .= sprintf( "Version: %s\n", $theme_info['Version'] );
		$output .= sprintf( "Description: %s\n", $theme_info['Description'] );
		$output .= sprintf( "Author: %s\n", $theme_info['Author'] );

		if ( ! empty( $theme_info['AuthorURI'] ) ) {
			$output .= sprintf( "Author URI: %s\n", $theme_info['AuthorURI'] );
		}

		if ( ! empty( $theme_info['ThemeURI'] ) ) {
			$output .= sprintf( "Theme URI: %s\n", $theme_info['ThemeURI'] );
		}

		$output .= sprintf( "Status: %s\n", $theme_info['Status'] );
		$output .= sprintf( "Text Domain: %s\n", $theme_info['TextDomain'] );

		if ( ! empty( $theme_info['DomainPath'] ) ) {
			$output .= sprintf( "Domain Path: %s\n", $theme_info['DomainPath'] );
		}

		if ( $is_child_theme ) {
			$output .= sprintf( "\nThis is a child theme of: %s\n", $theme_info['Template'] );
		}

		if ( ! empty( $theme_info['Tags'] ) && is_array( $theme_info['Tags'] ) ) {
			$output .= sprintf( "\nTags: %s\n", implode( ', ', $theme_info['Tags'] ) );
		}

		// Add theme directory path.
		$output .= sprintf( "\nTheme Directory: %s\n", $theme->get_stylesheet_directory() );
		$output .= sprintf( "Theme URL: %s\n", $theme->get_stylesheet_directory_uri() );

		return self::success_response( $output );
	}

	/**
	 * Create success response
	 *
	 * @param string $message Success message.
	 * @return array
	 */
	private static function success_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Create error response
	 *
	 * @param string $message Error message.
	 * @return array
	 */
	private static function error_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => 'Error: ' . $message,
				),
			),
			'isError' => true,
		);
	}
}
