<?php
/**
 * MCP Tool: Post Management
 *
 * Handles create, update, and delete operations for WordPress posts.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Post Tool Class
 */
class WebSamurai_MCP_Post_Tool {

	/**
	 * Get tool definition
	 *
	 * @return array
	 */
	public static function get_definition() {
		return array(
			'name'        => 'post_manage',
			'description' => 'Create, update, delete, or list WordPress posts',
			'inputSchema' => array(
				'type'       => 'object',
				'properties' => array(
					'action'         => array(
						'type'        => 'string',
						'description' => 'The action to perform',
						'enum'        => array( 'create', 'update', 'delete', 'list', 'get' ),
					),
					'id'             => array(
						'type'        => 'integer',
						'description' => 'Post ID (required for update and delete)',
					),
					'title'          => array(
						'type'        => 'string',
						'description' => 'Post title',
					),
					'content'        => array(
						'type'        => 'string',
						'description' => 'Post content',
					),
					'excerpt'        => array(
						'type'        => 'string',
						'description' => 'Post excerpt',
					),
					'status'         => array(
						'type'        => 'string',
						'description' => 'Post status (publish, draft, private, pending)',
						'enum'        => array( 'publish', 'draft', 'private', 'pending' ),
					),
					'categories'     => array(
						'type'        => 'array',
						'description' => 'Array of category IDs or names',
						'items'       => array(
							'oneOf' => array(
								array( 'type' => 'integer' ),
								array( 'type' => 'string' ),
							),
						),
					),
					'tags'           => array(
						'type'        => 'array',
						'description' => 'Array of tag IDs or names',
						'items'       => array(
							'oneOf' => array(
								array( 'type' => 'integer' ),
								array( 'type' => 'string' ),
							),
						),
					),
					'author'         => array(
						'type'        => 'integer',
						'description' => 'Author user ID',
					),
					'featured_image' => array(
						'type'        => 'integer',
						'description' => 'Featured image attachment ID',
					),
				),
				'required'   => array( 'action' ),
			),
		);
	}

	/**
	 * Execute the tool
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	public static function execute( $arguments ) {
		$action = isset( $arguments['action'] ) ? $arguments['action'] : '';

		switch ( $action ) {
			case 'create':
				return self::create_post( $arguments );

			case 'update':
				return self::update_post( $arguments );

			case 'delete':
				return self::delete_post( $arguments );

			case 'list':
				return self::list_posts( $arguments );

			case 'get':
				return self::get_post( $arguments );

			default:
				return self::error_response( 'Invalid action. Must be create, update, delete, list, or get.' );
		}
	}

	/**
	 * Create a new post
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function create_post( $arguments ) {
		if ( empty( $arguments['title'] ) ) {
			return self::error_response( 'Title is required for creating a post.' );
		}

		$post_data = array(
			'post_type'   => 'post',
			'post_title'  => sanitize_text_field( $arguments['title'] ),
			'post_status' => isset( $arguments['status'] ) ? sanitize_text_field( $arguments['status'] ) : 'draft',
		);

		if ( isset( $arguments['content'] ) ) {
			$post_data['post_content'] = wp_kses_post( $arguments['content'] );
		}

		if ( isset( $arguments['excerpt'] ) ) {
			$post_data['post_excerpt'] = sanitize_textarea_field( $arguments['excerpt'] );
		}

		if ( isset( $arguments['author'] ) ) {
			$post_data['post_author'] = absint( $arguments['author'] );
		}

		$post_id = wp_insert_post( $post_data );

		if ( is_wp_error( $post_id ) ) {
			return self::error_response( 'Failed to create post: ' . $post_id->get_error_message() );
		}

		// Set categories.
		if ( isset( $arguments['categories'] ) && is_array( $arguments['categories'] ) ) {
			self::set_categories( $post_id, $arguments['categories'] );
		}

		// Set tags.
		if ( isset( $arguments['tags'] ) && is_array( $arguments['tags'] ) ) {
			wp_set_post_tags( $post_id, $arguments['tags'] );
		}

		// Set featured image.
		if ( isset( $arguments['featured_image'] ) ) {
			set_post_thumbnail( $post_id, absint( $arguments['featured_image'] ) );
		}

		$post = get_post( $post_id );

		return self::success_response(
			sprintf(
				'Post created successfully. ID: %d, Title: "%s", Status: %s, URL: %s',
				$post_id,
				$post->post_title,
				$post->post_status,
				get_permalink( $post_id )
			)
		);
	}

	/**
	 * Update an existing post
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function update_post( $arguments ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Post ID is required for updating.' );
		}

		$post_id = absint( $arguments['id'] );
		$post    = get_post( $post_id );

		if ( ! $post || 'post' !== $post->post_type ) {
			return self::error_response( 'Post not found with ID: ' . $post_id );
		}

		$post_data = array( 'ID' => $post_id );

		if ( isset( $arguments['title'] ) ) {
			$post_data['post_title'] = sanitize_text_field( $arguments['title'] );
		}

		if ( isset( $arguments['content'] ) ) {
			$post_data['post_content'] = wp_kses_post( $arguments['content'] );
		}

		if ( isset( $arguments['excerpt'] ) ) {
			$post_data['post_excerpt'] = sanitize_textarea_field( $arguments['excerpt'] );
		}

		if ( isset( $arguments['status'] ) ) {
			$post_data['post_status'] = sanitize_text_field( $arguments['status'] );
		}

		if ( isset( $arguments['author'] ) ) {
			$post_data['post_author'] = absint( $arguments['author'] );
		}

		$result = wp_update_post( $post_data, true );

		if ( is_wp_error( $result ) ) {
			return self::error_response( 'Failed to update post: ' . $result->get_error_message() );
		}

		// Update categories.
		if ( isset( $arguments['categories'] ) && is_array( $arguments['categories'] ) ) {
			self::set_categories( $post_id, $arguments['categories'] );
		}

		// Update tags.
		if ( isset( $arguments['tags'] ) && is_array( $arguments['tags'] ) ) {
			wp_set_post_tags( $post_id, $arguments['tags'] );
		}

		// Update featured image.
		if ( isset( $arguments['featured_image'] ) ) {
			set_post_thumbnail( $post_id, absint( $arguments['featured_image'] ) );
		}

		$updated_post = get_post( $post_id );

		return self::success_response(
			sprintf(
				'Post updated successfully. ID: %d, Title: "%s", Status: %s, URL: %s',
				$post_id,
				$updated_post->post_title,
				$updated_post->post_status,
				get_permalink( $post_id )
			)
		);
	}

	/**
	 * Delete a post
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function delete_post( $arguments ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Post ID is required for deletion.' );
		}

		$post_id = absint( $arguments['id'] );
		$post    = get_post( $post_id );

		if ( ! $post || 'post' !== $post->post_type ) {
			return self::error_response( 'Post not found with ID: ' . $post_id );
		}

		$title  = $post->post_title;
		$result = wp_delete_post( $post_id, true );

		if ( ! $result ) {
			return self::error_response( 'Failed to delete post with ID: ' . $post_id );
		}

		return self::success_response(
			sprintf(
				'Post deleted successfully. ID: %d, Title: "%s"',
				$post_id,
				$title
			)
		);
	}

	/**
	 * Get a single post
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function get_post( $arguments ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Post ID is required for getting a post.' );
		}

		$post_id = absint( $arguments['id'] );
		$post    = get_post( $post_id );

		if ( ! $post || 'post' !== $post->post_type ) {
			return self::error_response( 'Post not found with ID: ' . $post_id );
		}

		$categories = get_the_category( $post_id );
		$cat_names  = ! empty( $categories ) ? implode( ', ', wp_list_pluck( $categories, 'name' ) ) : 'None';

		$tags      = get_the_tags( $post_id );
		$tag_names = ! empty( $tags ) ? implode( ', ', wp_list_pluck( $tags, 'name' ) ) : 'None';

		$output  = "Post Details:\n\n";
		$output .= 'ID: ' . $post->ID . "\n";
		$output .= 'Title: ' . $post->post_title . "\n";
		$output .= 'Status: ' . $post->post_status . "\n";
		$output .= 'Author: ' . get_the_author_meta( 'display_name', $post->post_author ) . ' (ID: ' . $post->post_author . ")\n";
		$output .= 'Date: ' . get_the_date( 'Y-m-d H:i:s', $post_id ) . "\n";
		$output .= 'Modified: ' . get_the_modified_date( 'Y-m-d H:i:s', $post_id ) . "\n";
		$output .= 'Categories: ' . $cat_names . "\n";
		$output .= 'Tags: ' . $tag_names . "\n";
		$output .= 'URL: ' . get_permalink( $post_id ) . "\n\n";
		$output .= "Excerpt:\n" . ( $post->post_excerpt ? $post->post_excerpt : '(none)' ) . "\n\n";
		$output .= "Content:\n" . $post->post_content;

		return self::success_response( $output );
	}

	/**
	 * List posts
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function list_posts( $arguments ) {
		$args = array(
			'post_type'      => 'post',
			'post_status'    => 'any',
			'posts_per_page' => is_array( $arguments ) && isset( $arguments['posts_per_page'] ) ? absint( $arguments['posts_per_page'] ) : -1,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		$posts = get_posts( $args );

		if ( empty( $posts ) ) {
			return self::success_response( 'No posts found.' );
		}

		$output = 'Found ' . count( $posts ) . " post(s):\n\n";

		foreach ( $posts as $post ) {
			$categories = get_the_category( $post->ID );
			$cat_names  = ! empty( $categories ) ? implode( ', ', wp_list_pluck( $categories, 'name' ) ) : 'None';

			$tags      = get_the_tags( $post->ID );
			$tag_names = ! empty( $tags ) ? implode( ', ', wp_list_pluck( $tags, 'name' ) ) : 'None';

			$output .= sprintf(
				"ID: %d | Title: \"%s\" | Status: %s | Categories: %s | Tags: %s | Date: %s | URL: %s\n",
				$post->ID,
				$post->post_title,
				$post->post_status,
				$cat_names,
				$tag_names,
				get_the_date( 'Y-m-d H:i:s', $post->ID ),
				get_permalink( $post->ID )
			);
		}

		return self::success_response( $output );
	}

	/**
	 * Set categories for a post
	 *
	 * @param int   $post_id Post ID.
	 * @param array $categories Array of category IDs or names.
	 * @return void
	 */
	private static function set_categories( $post_id, $categories ) {
		$category_ids = array();

		foreach ( $categories as $category ) {
			if ( is_numeric( $category ) ) {
				$category_ids[] = absint( $category );
			} else {
				$term = get_term_by( 'name', $category, 'category' );
				if ( ! $term ) {
					$term = wp_insert_term( $category, 'category' );
					if ( ! is_wp_error( $term ) ) {
						$category_ids[] = $term['term_id'];
					}
				} else {
					$category_ids[] = $term->term_id;
				}
			}
		}

		if ( ! empty( $category_ids ) ) {
			wp_set_post_categories( $post_id, $category_ids );
		}
	}

	/**
	 * Create success response
	 *
	 * @param string $message Success message.
	 * @return array
	 */
	private static function success_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Create error response
	 *
	 * @param string $message Error message.
	 * @return array
	 */
	private static function error_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => 'Error: ' . $message,
				),
			),
			'isError' => true,
		);
	}
}
