<?php
/**
 * MCP Tool: Plugin List
 *
 * Reads information about installed WordPress plugins.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plugin List Tool Class
 */
class WebSamurai_MCP_Plugin_List_Tool {

	/**
	 * Get tool definition
	 *
	 * @return array
	 */
	public static function get_definition() {
		return array(
			'name'        => 'plugin_list',
			'description' => 'Read information about installed WordPress plugins including name, version, description, author, and active status',
			'inputSchema' => array(
				'type'       => 'object',
				'properties' => array(
					'status' => array(
						'type'        => 'string',
						'description' => 'Filter plugins by status: all, active, inactive, or mu (must-use)',
						'enum'        => array( 'all', 'active', 'inactive', 'mu' ),
					),
				),
				'required'   => array(),
			),
		);
	}

	/**
	 * Execute the tool
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	public static function execute( $arguments ) {
		return self::get_plugin_list( $arguments );
	}

	/**
	 * Get plugin list
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function get_plugin_list( $arguments ) {
		$status = isset( $arguments['status'] ) ? sanitize_text_field( $arguments['status'] ) : 'all';

		// Ensure we have the plugin functions available.
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$output = '';

		// Handle must-use plugins separately.
		if ( 'mu' === $status ) {
			return self::get_mu_plugins();
		}

		// Get all installed plugins.
		$all_plugins    = get_plugins();
		$active_plugins = get_option( 'active_plugins', array() );

		// For multisite, also check network active plugins.
		if ( is_multisite() ) {
			$network_active = get_site_option( 'active_sitewide_plugins', array() );
		} else {
			$network_active = array();
		}

		// Filter plugins based on status.
		$filtered_plugins = array();
		foreach ( $all_plugins as $plugin_file => $plugin_data ) {
			$is_active         = in_array( $plugin_file, $active_plugins, true );
			$is_network_active = isset( $network_active[ $plugin_file ] );

			if ( 'active' === $status && ! $is_active && ! $is_network_active ) {
				continue;
			}

			if ( 'inactive' === $status && ( $is_active || $is_network_active ) ) {
				continue;
			}

			$filtered_plugins[ $plugin_file ]                      = $plugin_data;
			$filtered_plugins[ $plugin_file ]['is_active']         = $is_active || $is_network_active;
			$filtered_plugins[ $plugin_file ]['is_network_active'] = $is_network_active;
		}

		// Build output.
		$total_count  = count( $filtered_plugins );
		$active_count = count(
			array_filter(
				$filtered_plugins,
				function ( $plugin ) {
					return $plugin['is_active'];
				}
			)
		);

		$output .= "Installed Plugins\n";
		$output .= str_repeat( '=', 50 ) . "\n\n";
		$output .= sprintf( 'Total: %d plugin(s)', $total_count );

		if ( 'all' === $status ) {
			$output .= sprintf( " (%d active, %d inactive)\n\n", $active_count, $total_count - $active_count );
		} else {
			$output .= "\n\n";
		}

		if ( empty( $filtered_plugins ) ) {
			$output .= sprintf( "No %s plugins found.\n", 'all' === $status ? 'installed' : $status );
			return self::success_response( $output );
		}

		// List each plugin.
		foreach ( $filtered_plugins as $plugin_file => $plugin_data ) {
			$status_label = $plugin_data['is_network_active'] ? '[Network Active]' : ( $plugin_data['is_active'] ? '[Active]' : '[Inactive]' );

			$output .= sprintf( "%s %s\n", $status_label, $plugin_data['Name'] );
			$output .= sprintf( "  Version: %s\n", $plugin_data['Version'] );

			if ( ! empty( $plugin_data['Description'] ) ) {
				$output .= sprintf( "  Description: %s\n", $plugin_data['Description'] );
			}

			if ( ! empty( $plugin_data['Author'] ) ) {
				$output .= sprintf( "  Author: %s\n", wp_strip_all_tags( $plugin_data['Author'] ) );
			}

			if ( ! empty( $plugin_data['PluginURI'] ) ) {
				$output .= sprintf( "  Plugin URI: %s\n", $plugin_data['PluginURI'] );
			}

			if ( ! empty( $plugin_data['TextDomain'] ) ) {
				$output .= sprintf( "  Text Domain: %s\n", $plugin_data['TextDomain'] );
			}

			$output .= sprintf( "  File: %s\n", $plugin_file );
			$output .= "\n";
		}

		return self::success_response( $output );
	}

	/**
	 * Get must-use plugins
	 *
	 * @return array
	 */
	private static function get_mu_plugins() {
		// Ensure we have the plugin functions available.
		if ( ! function_exists( 'get_mu_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$mu_plugins = get_mu_plugins();

		$output  = "Must-Use Plugins\n";
		$output .= str_repeat( '=', 50 ) . "\n\n";
		$output .= sprintf( "Total: %d plugin(s)\n\n", count( $mu_plugins ) );

		if ( empty( $mu_plugins ) ) {
			$output .= "No must-use plugins found.\n";
			return self::success_response( $output );
		}

		foreach ( $mu_plugins as $plugin_file => $plugin_data ) {
			$output .= sprintf( "[Must-Use] %s\n", $plugin_data['Name'] );
			$output .= sprintf( "  Version: %s\n", $plugin_data['Version'] );

			if ( ! empty( $plugin_data['Description'] ) ) {
				$output .= sprintf( "  Description: %s\n", $plugin_data['Description'] );
			}

			if ( ! empty( $plugin_data['Author'] ) ) {
				$output .= sprintf( "  Author: %s\n", wp_strip_all_tags( $plugin_data['Author'] ) );
			}

			$output .= sprintf( "  File: %s\n", $plugin_file );
			$output .= "\n";
		}

		return self::success_response( $output );
	}

	/**
	 * Create success response
	 *
	 * @param string $message Success message.
	 * @return array
	 */
	private static function success_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Create error response
	 *
	 * @param string $message Error message.
	 * @return array
	 */
	private static function error_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => 'Error: ' . $message,
				),
			),
			'isError' => true,
		);
	}
}
