<?php
/**
 * MCP Tool: Page Management
 *
 * Handles create, update, and delete operations for WordPress pages.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Page Tool Class
 */
class WebSamurai_MCP_Page_Tool {

	/**
	 * Get tool definition
	 *
	 * @return array
	 */
	public static function get_definition() {
		return array(
			'name'        => 'page_manage',
			'description' => 'Create, update, delete, or list WordPress pages',
			'inputSchema' => array(
				'type'       => 'object',
				'properties' => array(
					'action'     => array(
						'type'        => 'string',
						'description' => 'The action to perform',
						'enum'        => array( 'create', 'update', 'delete', 'list', 'get' ),
					),
					'id'         => array(
						'type'        => 'integer',
						'description' => 'Page ID (required for update and delete)',
					),
					'title'      => array(
						'type'        => 'string',
						'description' => 'Page title',
					),
					'content'    => array(
						'type'        => 'string',
						'description' => 'Page content',
					),
					'status'     => array(
						'type'        => 'string',
						'description' => 'Page status (publish, draft, private, pending)',
						'enum'        => array( 'publish', 'draft', 'private', 'pending' ),
					),
					'parent'     => array(
						'type'        => 'integer',
						'description' => 'Parent page ID',
					),
					'menu_order' => array(
						'type'        => 'integer',
						'description' => 'Page order in menu',
					),
					'template'   => array(
						'type'        => 'string',
						'description' => 'Page template filename',
					),
				),
				'required'   => array( 'action' ),
			),
		);
	}

	/**
	 * Execute the tool
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	public static function execute( $arguments ) {
		$action = isset( $arguments['action'] ) ? $arguments['action'] : '';

		switch ( $action ) {
			case 'create':
				return self::create_page( $arguments );

			case 'update':
				return self::update_page( $arguments );

			case 'delete':
				return self::delete_page( $arguments );

			case 'list':
				return self::list_pages( $arguments );

			case 'get':
				return self::get_page( $arguments );

			default:
				return self::error_response( 'Invalid action. Must be create, update, delete, list, or get.' );
		}
	}

	/**
	 * Create a new page
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function create_page( $arguments ) {
		if ( empty( $arguments['title'] ) ) {
			return self::error_response( 'Title is required for creating a page.' );
		}

		$page_data = array(
			'post_type'   => 'page',
			'post_title'  => sanitize_text_field( $arguments['title'] ),
			'post_status' => isset( $arguments['status'] ) ? sanitize_text_field( $arguments['status'] ) : 'draft',
		);

		if ( isset( $arguments['content'] ) ) {
			$page_data['post_content'] = wp_kses_post( $arguments['content'] );
		}

		if ( isset( $arguments['parent'] ) ) {
			$page_data['post_parent'] = absint( $arguments['parent'] );
		}

		if ( isset( $arguments['menu_order'] ) ) {
			$page_data['menu_order'] = absint( $arguments['menu_order'] );
		}

		$page_id = wp_insert_post( $page_data );

		if ( is_wp_error( $page_id ) ) {
			return self::error_response( 'Failed to create page: ' . $page_id->get_error_message() );
		}

		// Set page template if provided.
		if ( isset( $arguments['template'] ) ) {
			update_post_meta( $page_id, '_wp_page_template', sanitize_text_field( $arguments['template'] ) );
		}

		$page = get_post( $page_id );

		return self::success_response(
			sprintf(
				'Page created successfully. ID: %d, Title: "%s", Status: %s, URL: %s',
				$page_id,
				$page->post_title,
				$page->post_status,
				get_permalink( $page_id )
			)
		);
	}

	/**
	 * Update an existing page
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function update_page( $arguments ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Page ID is required for updating.' );
		}

		$page_id = absint( $arguments['id'] );
		$page    = get_post( $page_id );

		if ( ! $page || 'page' !== $page->post_type ) {
			return self::error_response( 'Page not found with ID: ' . $page_id );
		}

		$page_data = array( 'ID' => $page_id );

		if ( isset( $arguments['title'] ) ) {
			$page_data['post_title'] = sanitize_text_field( $arguments['title'] );
		}

		if ( isset( $arguments['content'] ) ) {
			$page_data['post_content'] = wp_kses_post( $arguments['content'] );
		}

		if ( isset( $arguments['status'] ) ) {
			$page_data['post_status'] = sanitize_text_field( $arguments['status'] );
		}

		if ( isset( $arguments['parent'] ) ) {
			$page_data['post_parent'] = absint( $arguments['parent'] );
		}

		if ( isset( $arguments['menu_order'] ) ) {
			$page_data['menu_order'] = absint( $arguments['menu_order'] );
		}

		$result = wp_update_post( $page_data, true );

		if ( is_wp_error( $result ) ) {
			return self::error_response( 'Failed to update page: ' . $result->get_error_message() );
		}

		// Update page template if provided.
		if ( isset( $arguments['template'] ) ) {
			update_post_meta( $page_id, '_wp_page_template', sanitize_text_field( $arguments['template'] ) );
		}

		$updated_page = get_post( $page_id );

		return self::success_response(
			sprintf(
				'Page updated successfully. ID: %d, Title: "%s", Status: %s, URL: %s',
				$page_id,
				$updated_page->post_title,
				$updated_page->post_status,
				get_permalink( $page_id )
			)
		);
	}

	/**
	 * Delete a page
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function delete_page( $arguments ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Page ID is required for deletion.' );
		}

		$page_id = absint( $arguments['id'] );
		$page    = get_post( $page_id );

		if ( ! $page || 'page' !== $page->post_type ) {
			return self::error_response( 'Page not found with ID: ' . $page_id );
		}

		$title  = $page->post_title;
		$result = wp_delete_post( $page_id, true );

		if ( ! $result ) {
			return self::error_response( 'Failed to delete page with ID: ' . $page_id );
		}

		return self::success_response(
			sprintf(
				'Page deleted successfully. ID: %d, Title: "%s"',
				$page_id,
				$title
			)
		);
	}

	/**
	 * Get a single page
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function get_page( $arguments ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Page ID is required for getting a page.' );
		}

		$page_id = absint( $arguments['id'] );
		$page    = get_post( $page_id );

		if ( ! $page || 'page' !== $page->post_type ) {
			return self::error_response( 'Page not found with ID: ' . $page_id );
		}

		$output  = "Page Details:\n\n";
		$output .= 'ID: ' . $page->ID . "\n";
		$output .= 'Title: ' . $page->post_title . "\n";
		$output .= 'Status: ' . $page->post_status . "\n";
		$output .= 'Author: ' . get_the_author_meta( 'display_name', $page->post_author ) . ' (ID: ' . $page->post_author . ")\n";
		$output .= 'Date: ' . get_the_date( 'Y-m-d H:i:s', $page->ID ) . "\n";
		$output .= 'Modified: ' . get_the_modified_date( 'Y-m-d H:i:s', $page->ID ) . "\n";
		$output .= 'Parent: ' . ( $page->post_parent ? get_the_title( $page->post_parent ) . ' (' . $page->post_parent . ')' : 'None' ) . "\n";
		$output .= 'Template: ' . ( get_page_template_slug( $page->ID ) ? get_page_template_slug( $page->ID ) : 'Default' ) . "\n";
		$output .= 'URL: ' . get_permalink( $page->ID ) . "\n\n";
		$output .= "Content:\n" . $page->post_content;

		return self::success_response( $output );
	}

	/**
	 * List pages
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function list_pages( $arguments = array() ) {
		$args = array(
			'post_type'      => 'page',
			'post_status'    => 'any',
			'posts_per_page' => is_array( $arguments ) && isset( $arguments['posts_per_page'] ) ? absint( $arguments['posts_per_page'] ) : -1,
			'orderby'        => 'title',
			'order'          => 'ASC',
		);

		$pages = get_posts( $args );

		if ( empty( $pages ) ) {
			return self::success_response( 'No pages found.' );
		}

		$output = 'Found ' . count( $pages ) . " page(s):\n\n";

		foreach ( $pages as $page ) {
			$output .= sprintf(
				"ID: %d | Title: \"%s\" | Status: %s | Parent: %s | URL: %s\n",
				$page->ID,
				$page->post_title,
				$page->post_status,
				$page->post_parent ? get_the_title( $page->post_parent ) . ' (' . $page->post_parent . ')' : 'None',
				get_permalink( $page->ID )
			);
		}

		return self::success_response( $output );
	}

	/**
	 * Create success response
	 *
	 * @param string $message Success message.
	 * @return array
	 */
	private static function success_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Create error response
	 *
	 * @param string $message Error message.
	 * @return array
	 */
	private static function error_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => 'Error: ' . $message,
				),
			),
			'isError' => true,
		);
	}
}
