<?php
/**
 * MCP Tool: Custom Post Type Management
 *
 * Handles create, update, and delete operations for custom post types.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Custom Post Tool Class
 */
class WebSamurai_MCP_Custom_Post_Tool {

	/**
	 * Get tool definition
	 *
	 * @return array
	 */
	public static function get_definition() {
		return array(
			'name'        => 'custom_post_manage',
			'description' => 'Create, update, delete, or list custom post type entries, or list all available custom post types',
			'inputSchema' => array(
				'type'       => 'object',
				'properties' => array(
					'action'         => array(
						'type'        => 'string',
						'description' => 'The action to perform',
						'enum'        => array( 'create', 'update', 'delete', 'list', 'get', 'list_types' ),
					),
					'post_type'      => array(
						'type'        => 'string',
						'description' => 'The custom post type slug (e.g., "product", "event"). Not required for list_types action.',
					),
					'id'             => array(
						'type'        => 'integer',
						'description' => 'Post ID (required for update and delete)',
					),
					'title'          => array(
						'type'        => 'string',
						'description' => 'Post title',
					),
					'content'        => array(
						'type'        => 'string',
						'description' => 'Post content',
					),
					'excerpt'        => array(
						'type'        => 'string',
						'description' => 'Post excerpt',
					),
					'status'         => array(
						'type'        => 'string',
						'description' => 'Post status (publish, draft, private, pending)',
						'enum'        => array( 'publish', 'draft', 'private', 'pending' ),
					),
					'author'         => array(
						'type'        => 'integer',
						'description' => 'Author user ID',
					),
					'meta_data'      => array(
						'type'                 => 'object',
						'description'          => 'Custom meta fields as key-value pairs',
						'additionalProperties' => true,
					),
					'taxonomies'     => array(
						'type'                 => 'object',
						'description'          => 'Taxonomies and their terms (e.g., {"product_cat": ["electronics"], "product_tag": ["featured"]})',
						'additionalProperties' => array(
							'type'  => 'array',
							'items' => array(
								'oneOf' => array(
									array( 'type' => 'integer' ),
									array( 'type' => 'string' ),
								),
							),
						),
					),
					'featured_image' => array(
						'type'        => 'integer',
						'description' => 'Featured image attachment ID',
					),
				),
				'required'   => array( 'action' ),
			),
		);
	}

	/**
	 * Execute the tool
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	public static function execute( $arguments ) {
		$action = isset( $arguments['action'] ) ? $arguments['action'] : '';

		// Handle list_types action separately (doesn't require post_type).
		if ( 'list_types' === $action ) {
			return self::list_types();
		}

		// Validate post type for all other actions.
		$post_type = isset( $arguments['post_type'] ) ? sanitize_key( $arguments['post_type'] ) : '';

		if ( empty( $post_type ) ) {
			return self::error_response( 'Post type is required.' );
		}

		if ( ! post_type_exists( $post_type ) ) {
			return self::error_response( 'Post type "' . $post_type . '" does not exist.' );
		}

		// Exclude built-in post types (handled by dedicated tools).
		if ( in_array( $post_type, array( 'post', 'page', 'attachment', 'revision', 'nav_menu_item' ), true ) ) {
			return self::error_response( 'Use dedicated tools for built-in post types (post, page).' );
		}

		switch ( $action ) {
			case 'create':
				return self::create_post( $arguments, $post_type );

			case 'update':
				return self::update_post( $arguments, $post_type );

			case 'delete':
				return self::delete_post( $arguments, $post_type );

			case 'list':
				return self::list_posts( $arguments, $post_type );

			case 'get':
				return self::get_post( $arguments, $post_type );

			default:
				return self::error_response( 'Invalid action. Must be create, update, delete, list, get, or list_types.' );
		}
	}

	/**
	 * Create a new custom post
	 *
	 * @param array  $arguments Tool arguments.
	 * @param string $post_type Post type slug.
	 * @return array
	 */
	private static function create_post( $arguments, $post_type ) {
		if ( empty( $arguments['title'] ) ) {
			return self::error_response( 'Title is required for creating a post.' );
		}

		$post_data = array(
			'post_type'   => $post_type,
			'post_title'  => sanitize_text_field( $arguments['title'] ),
			'post_status' => isset( $arguments['status'] ) ? sanitize_text_field( $arguments['status'] ) : 'draft',
		);

		if ( isset( $arguments['content'] ) ) {
			$post_data['post_content'] = wp_kses_post( $arguments['content'] );
		}

		if ( isset( $arguments['excerpt'] ) ) {
			$post_data['post_excerpt'] = sanitize_textarea_field( $arguments['excerpt'] );
		}

		if ( isset( $arguments['author'] ) ) {
			$post_data['post_author'] = absint( $arguments['author'] );
		}

		$post_id = wp_insert_post( $post_data );

		if ( is_wp_error( $post_id ) ) {
			return self::error_response( 'Failed to create post: ' . $post_id->get_error_message() );
		}

		// Set custom meta data.
		if ( isset( $arguments['meta_data'] ) && is_array( $arguments['meta_data'] ) ) {
			self::set_meta_data( $post_id, $arguments['meta_data'] );
		}

		// Set taxonomies.
		if ( isset( $arguments['taxonomies'] ) && is_array( $arguments['taxonomies'] ) ) {
			self::set_taxonomies( $post_id, $arguments['taxonomies'] );
		}

		// Set featured image.
		if ( isset( $arguments['featured_image'] ) ) {
			set_post_thumbnail( $post_id, absint( $arguments['featured_image'] ) );
		}

		$post          = get_post( $post_id );
		$post_type_obj = get_post_type_object( $post_type );

		return self::success_response(
			sprintf(
				'%s created successfully. ID: %d, Title: "%s", Status: %s, URL: %s',
				$post_type_obj->labels->singular_name,
				$post_id,
				$post->post_title,
				$post->post_status,
				get_permalink( $post_id )
			)
		);
	}

	/**
	 * Update an existing custom post
	 *
	 * @param array  $arguments Tool arguments.
	 * @param string $post_type Post type slug.
	 * @return array
	 */
	private static function update_post( $arguments, $post_type ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Post ID is required for updating.' );
		}

		$post_id = absint( $arguments['id'] );
		$post    = get_post( $post_id );

		if ( ! $post || $post_type !== $post->post_type ) {
			return self::error_response( 'Post not found with ID: ' . $post_id . ' and type: ' . $post_type );
		}

		$post_data = array( 'ID' => $post_id );

		if ( isset( $arguments['title'] ) ) {
			$post_data['post_title'] = sanitize_text_field( $arguments['title'] );
		}

		if ( isset( $arguments['content'] ) ) {
			$post_data['post_content'] = wp_kses_post( $arguments['content'] );
		}

		if ( isset( $arguments['excerpt'] ) ) {
			$post_data['post_excerpt'] = sanitize_textarea_field( $arguments['excerpt'] );
		}

		if ( isset( $arguments['status'] ) ) {
			$post_data['post_status'] = sanitize_text_field( $arguments['status'] );
		}

		if ( isset( $arguments['author'] ) ) {
			$post_data['post_author'] = absint( $arguments['author'] );
		}

		$result = wp_update_post( $post_data, true );

		if ( is_wp_error( $result ) ) {
			return self::error_response( 'Failed to update post: ' . $result->get_error_message() );
		}

		// Update custom meta data.
		if ( isset( $arguments['meta_data'] ) && is_array( $arguments['meta_data'] ) ) {
			self::set_meta_data( $post_id, $arguments['meta_data'] );
		}

		// Update taxonomies.
		if ( isset( $arguments['taxonomies'] ) && is_array( $arguments['taxonomies'] ) ) {
			self::set_taxonomies( $post_id, $arguments['taxonomies'] );
		}

		// Update featured image.
		if ( isset( $arguments['featured_image'] ) ) {
			set_post_thumbnail( $post_id, absint( $arguments['featured_image'] ) );
		}

		$updated_post  = get_post( $post_id );
		$post_type_obj = get_post_type_object( $post_type );

		return self::success_response(
			sprintf(
				'%s updated successfully. ID: %d, Title: "%s", Status: %s, URL: %s',
				$post_type_obj->labels->singular_name,
				$post_id,
				$updated_post->post_title,
				$updated_post->post_status,
				get_permalink( $post_id )
			)
		);
	}

	/**
	 * Delete a custom post
	 *
	 * @param array  $arguments Tool arguments.
	 * @param string $post_type Post type slug.
	 * @return array
	 */
	private static function delete_post( $arguments, $post_type ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Post ID is required for deletion.' );
		}

		$post_id = absint( $arguments['id'] );
		$post    = get_post( $post_id );

		if ( ! $post || $post_type !== $post->post_type ) {
			return self::error_response( 'Post not found with ID: ' . $post_id . ' and type: ' . $post_type );
		}

		$title         = $post->post_title;
		$post_type_obj = get_post_type_object( $post_type );
		$result        = wp_delete_post( $post_id, true );

		if ( ! $result ) {
			return self::error_response( 'Failed to delete post with ID: ' . $post_id );
		}

		return self::success_response(
			sprintf(
				'%s deleted successfully. ID: %d, Title: "%s"',
				$post_type_obj->labels->singular_name,
				$post_id,
				$title
			)
		);
	}

	/**
	 * Get a single custom post
	 *
	 * @param array  $arguments Tool arguments.
	 * @param string $post_type Post type slug.
	 * @return array
	 */
	private static function get_post( $arguments, $post_type ) {
		if ( empty( $arguments['id'] ) ) {
			return self::error_response( 'Post ID is required for getting a post.' );
		}

		$post_id = absint( $arguments['id'] );
		$post    = get_post( $post_id );

		if ( ! $post || $post_type !== $post->post_type ) {
			return self::error_response( 'Post not found with ID: ' . $post_id . ' and type: ' . $post_type );
		}

		$post_type_obj = get_post_type_object( $post_type );

		// Get taxonomies for this post type.
		$taxonomies = get_object_taxonomies( $post_type, 'objects' );
		$tax_info   = array();

		foreach ( $taxonomies as $taxonomy ) {
			$terms = get_the_terms( $post_id, $taxonomy->name );
			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
				$tax_info[] = $taxonomy->label . ': ' . implode( ', ', wp_list_pluck( $terms, 'name' ) );
			}
		}

		$tax_display = ! empty( $tax_info ) ? implode( "\n", $tax_info ) : 'None';

		// Get custom meta data.
		$meta_data    = get_post_meta( $post_id );
		$meta_display = '';
		if ( ! empty( $meta_data ) ) {
			foreach ( $meta_data as $key => $values ) {
				// Skip internal WordPress meta.
				if ( strpos( $key, '_' ) === 0 ) {
					continue;
				}
				$meta_display .= $key . ': ' . ( is_array( $values ) ? implode( ', ', $values ) : $values ) . "\n";
			}
		}

		$output  = $post_type_obj->labels->singular_name . " Details:\n\n";
		$output .= 'ID: ' . $post->ID . "\n";
		$output .= 'Title: ' . $post->post_title . "\n";
		$output .= 'Status: ' . $post->post_status . "\n";
		$output .= 'Author: ' . get_the_author_meta( 'display_name', $post->post_author ) . ' (ID: ' . $post->post_author . ")\n";
		$output .= 'Date: ' . get_the_date( 'Y-m-d H:i:s', $post_id ) . "\n";
		$output .= 'Modified: ' . get_the_modified_date( 'Y-m-d H:i:s', $post_id ) . "\n";
		$output .= "Taxonomies:\n" . $tax_display . "\n";
		if ( $meta_display ) {
			$output .= "Custom Fields:\n" . $meta_display;
		}
		$output .= 'URL: ' . get_permalink( $post_id ) . "\n\n";
		$output .= "Excerpt:\n" . ( $post->post_excerpt ? $post->post_excerpt : '(none)' ) . "\n\n";
		$output .= "Content:\n" . $post->post_content;

		return self::success_response( $output );
	}

	/**
	 * List custom post type entries
	 *
	 * @param array  $arguments Tool arguments.
	 * @param string $post_type Post type slug.
	 * @return array
	 */
	private static function list_posts( $arguments, $post_type ) {
		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'any',
			'posts_per_page' => -1,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		$posts = get_posts( $args );

		if ( empty( $posts ) ) {
			$post_type_obj = get_post_type_object( $post_type );
			return self::success_response( 'No ' . strtolower( $post_type_obj->labels->name ) . ' found.' );
		}

		$post_type_obj = get_post_type_object( $post_type );
		$output        = 'Found ' . count( $posts ) . ' ' . strtolower( $post_type_obj->labels->name ) . ":\n\n";

		foreach ( $posts as $post ) {
			// Get taxonomies for this post type.
			$taxonomies = get_object_taxonomies( $post_type, 'objects' );
			$tax_info   = array();

			foreach ( $taxonomies as $taxonomy ) {
				$terms = get_the_terms( $post->ID, $taxonomy->name );
				if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
					$tax_info[] = $taxonomy->label . ': ' . implode( ', ', wp_list_pluck( $terms, 'name' ) );
				}
			}

			$tax_display = ! empty( $tax_info ) ? implode( ' | ', $tax_info ) : 'None';

			$output .= sprintf(
				"ID: %d | Title: \"%s\" | Status: %s | Taxonomies: %s | Date: %s | URL: %s\n",
				$post->ID,
				$post->post_title,
				$post->post_status,
				$tax_display,
				get_the_date( 'Y-m-d H:i:s', $post->ID ),
				get_permalink( $post->ID )
			);
		}

		return self::success_response( $output );
	}

	/**
	 * List all available custom post types
	 *
	 * @return array
	 */
	private static function list_types() {
		// Get all registered post types.
		$args = array(
			'public'   => true,
			'_builtin' => false,
		);

		$post_types = get_post_types( $args, 'objects' );

		if ( empty( $post_types ) ) {
			return self::success_response( 'No custom post types found.' );
		}

		$output = 'Found ' . count( $post_types ) . " custom post type(s):\n\n";

		foreach ( $post_types as $post_type ) {
			// Get post count.
			$count = wp_count_posts( $post_type->name );
			$total = 0;
			if ( $count ) {
				foreach ( $count as $status => $num ) {
					$total += $num;
				}
			}

			// Get associated taxonomies.
			$taxonomies = get_object_taxonomies( $post_type->name, 'objects' );
			$tax_names  = array();
			foreach ( $taxonomies as $taxonomy ) {
				$tax_names[] = $taxonomy->label . ' (' . $taxonomy->name . ')';
			}
			$tax_display = ! empty( $tax_names ) ? implode( ', ', $tax_names ) : 'None';

			$output .= sprintf(
				"Slug: %s\nName: %s (singular: %s)\nDescription: %s\nPost Count: %d\nTaxonomies: %s\nHierarchical: %s\nSupports: %s\n\n",
				$post_type->name,
				$post_type->label,
				$post_type->labels->singular_name,
				$post_type->description ? $post_type->description : 'N/A',
				$total,
				$tax_display,
				$post_type->hierarchical ? 'Yes' : 'No',
				implode( ', ', get_all_post_type_supports( $post_type->name ) )
			);
		}

		return self::success_response( $output );
	}

	/**
	 * Set meta data for a post
	 *
	 * @param int   $post_id Post ID.
	 * @param array $meta_data Meta data key-value pairs.
	 * @return void
	 */
	private static function set_meta_data( $post_id, $meta_data ) {
		foreach ( $meta_data as $key => $value ) {
			$sanitized_key = sanitize_key( $key );

			// Sanitize value based on type.
			if ( is_array( $value ) ) {
				$sanitized_value = array_map( 'sanitize_text_field', $value );
			} elseif ( is_numeric( $value ) ) {
				$sanitized_value = $value;
			} else {
				$sanitized_value = sanitize_text_field( $value );
			}

			update_post_meta( $post_id, $sanitized_key, $sanitized_value );
		}
	}

	/**
	 * Set taxonomies for a post
	 *
	 * @param int   $post_id Post ID.
	 * @param array $taxonomies Taxonomy terms.
	 * @return void
	 */
	private static function set_taxonomies( $post_id, $taxonomies ) {
		foreach ( $taxonomies as $taxonomy => $terms ) {
			$sanitized_taxonomy = sanitize_key( $taxonomy );

			if ( ! taxonomy_exists( $sanitized_taxonomy ) ) {
				continue;
			}

			$term_ids = array();

			foreach ( $terms as $term ) {
				if ( is_numeric( $term ) ) {
					$term_ids[] = absint( $term );
				} else {
					$term_obj = get_term_by( 'name', $term, $sanitized_taxonomy );
					if ( ! $term_obj ) {
						$term_obj = wp_insert_term( $term, $sanitized_taxonomy );
						if ( ! is_wp_error( $term_obj ) ) {
							$term_ids[] = $term_obj['term_id'];
						}
					} else {
						$term_ids[] = $term_obj->term_id;
					}
				}
			}

			if ( ! empty( $term_ids ) ) {
				wp_set_object_terms( $post_id, $term_ids, $sanitized_taxonomy );
			}
		}
	}

	/**
	 * Create success response
	 *
	 * @param string $message Success message.
	 * @return array
	 */
	private static function success_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Create error response
	 *
	 * @param string $message Error message.
	 * @return array
	 */
	private static function error_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => 'Error: ' . $message,
				),
			),
			'isError' => true,
		);
	}
}
