<?php
/**
 * MCP Tool: Additional CSS Management
 *
 * Handles reading and writing WordPress Customizer Additional CSS.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Additional CSS Tool Class
 */
class WebSamurai_MCP_Additional_CSS_Tool {

	/**
	 * Get tool definition
	 *
	 * @return array
	 */
	public static function get_definition() {
		return array(
			'name'        => 'additional_css_manage',
			'description' => 'Read or write the WordPress Customizer Additional CSS',
			'inputSchema' => array(
				'type'       => 'object',
				'properties' => array(
					'action'     => array(
						'type'        => 'string',
						'description' => 'The action to perform',
						'enum'        => array( 'get', 'update' ),
					),
					'css'        => array(
						'type'        => 'string',
						'description' => 'The CSS content to write (required for update action)',
					),
					'stylesheet' => array(
						'type'        => 'string',
						'description' => 'The stylesheet (theme) slug. Defaults to current theme.',
					),
				),
				'required'   => array( 'action' ),
			),
		);
	}

	/**
	 * Execute the tool
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	public static function execute( $arguments ) {
		$action = isset( $arguments['action'] ) ? $arguments['action'] : '';

		switch ( $action ) {
			case 'get':
				return self::get_css( $arguments );

			case 'update':
				return self::update_css( $arguments );

			default:
				return self::error_response( 'Invalid action. Must be get or update.' );
		}
	}

	/**
	 * Get the Additional CSS
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function get_css( $arguments ) {
		$stylesheet = isset( $arguments['stylesheet'] ) ? sanitize_text_field( $arguments['stylesheet'] ) : get_stylesheet();

		// Get the custom CSS for the specified stylesheet.
		$css = wp_get_custom_css( $stylesheet );

		if ( empty( $css ) ) {
			return self::success_response(
				sprintf(
					"No Additional CSS found for theme: %s\n\nThe Additional CSS is empty.",
					$stylesheet
				)
			);
		}

		$output = sprintf(
			"Additional CSS for theme: %s\n\nCSS Content:\n%s",
			$stylesheet,
			$css
		);

		return self::success_response( $output );
	}

	/**
	 * Update the Additional CSS
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private static function update_css( $arguments ) {
		if ( ! isset( $arguments['css'] ) ) {
			return self::error_response( 'CSS content is required for update action.' );
		}

		$css        = $arguments['css'];
		$stylesheet = isset( $arguments['stylesheet'] ) ? sanitize_text_field( $arguments['stylesheet'] ) : get_stylesheet();

		// Validate CSS (basic validation).
		$css = wp_strip_all_tags( $css );

		// Update the custom CSS.
		$result = wp_update_custom_css_post(
			$css
		);

		if ( is_wp_error( $result ) ) {
			return self::error_response( 'Failed to update Additional CSS: ' . $result->get_error_message() );
		}

		// Get the updated CSS to confirm.
		$updated_css = wp_get_custom_css( $stylesheet );

		return self::success_response(
			sprintf(
				"Additional CSS updated successfully for theme: %s\n\nCSS Length: %d characters\n\nUpdated CSS:\n%s",
				$stylesheet,
				strlen( $updated_css ),
				$updated_css
			)
		);
	}

	/**
	 * Create success response
	 *
	 * @param string $message Success message.
	 * @return array
	 */
	private static function success_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Create error response
	 *
	 * @param string $message Error message.
	 * @return array
	 */
	private static function error_response( $message ) {
		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => 'Error: ' . $message,
				),
			),
			'isError' => true,
		);
	}
}
