<?php
/**
 * Enqueue scripts and styles.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Enqueue admin scripts and styles.
 *
 * @return void
 */
function websamurai_enqueue_admin_assets() {
	// Only load on our admin page.

	$asset_file = WEBSAMURAI_PLUGIN_DIR . 'build/admin.asset.php';
	if ( file_exists( $asset_file ) ) {
		$asset = require $asset_file;

		wp_enqueue_script(
			'websamurai-admin',
			WEBSAMURAI_PLUGIN_URL . 'build/admin.js',
			$asset['dependencies'],
			$asset['version'],
			true
		);

		wp_enqueue_style(
			'websamurai-admin',
			WEBSAMURAI_PLUGIN_URL . 'build/admin.css',
			array(),
			$asset['version']
		);

		// Determine OAuth base URL based on LOCALDEV constant.
		$oauth_base_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500' : 'https://cp.websamurai.io';

		// Localize script with data.
		wp_localize_script(
			'websamurai-admin',
			'samurAiAdmin',
			array(
				'ajaxUrl'      => admin_url( 'admin-ajax.php' ),
				'nonce'        => wp_create_nonce( 'wp_rest' ),
				'apiUrl'       => rest_url( 'websamurai/v1' ),
				'oauthBaseUrl' => $oauth_base_url,
			)
		);
	}
}
add_action( 'admin_enqueue_scripts', 'websamurai_enqueue_admin_assets' );

/**
 * Enqueue frontend scripts and styles.
 *
 * @return void
 */
function websamurai_enqueue_frontend_assets() {
	// Only load for logged-in users who are editors or administrators.
	if ( ! is_user_logged_in() ) {
		return;
	}

	// Check if user is an editor or administrator.
	$user               = wp_get_current_user();
	$is_editor_or_admin = in_array( 'editor', $user->roles, true ) || in_array( 'administrator', $user->roles, true );

	if ( ! $is_editor_or_admin ) {
		return;
	}

	// Check if user is connected to OAuth service.
	$auth_key = get_option( 'websamurai_cp_auth_key' );
	if ( empty( $auth_key ) ) {
		// Not connected, don't load the frontend assets.
		return;
	}

	$asset_file = WEBSAMURAI_PLUGIN_DIR . 'build/frontend.asset.php';

	if ( file_exists( $asset_file ) ) {
		$asset = require $asset_file;

		wp_enqueue_script(
			'websamurai-frontend',
			WEBSAMURAI_PLUGIN_URL . 'build/frontend.js',
			$asset['dependencies'],
			$asset['version'],
			true
		);

		wp_enqueue_style(
			'websamurai-frontend',
			WEBSAMURAI_PLUGIN_URL . 'build/frontend.css',
			array(),
			$asset['version']
		);

		global $post;
		$location_type = '';
		if ( is_single() ) {
			$location_type = 'single';
		} elseif ( is_home() ) {
			$location_type = 'blog';
		} elseif ( is_archive() ) {
			$location_type = 'archive';
		} elseif ( is_front_page() ) {
			$location_type = 'frontpage';
		} elseif ( is_page() ) {
			$location_type = 'page';
		} elseif ( is_search() ) {
			$location_type = 'search';
		} elseif ( is_404() ) {
			$location_type = '404';
		} else {
			$location_type = 'other';
		}

		// Localize script with data.
		wp_localize_script(
			'websamurai-frontend',
			'samurAiFrontend',
			array(
				'ajaxUrl'      => admin_url( 'admin-ajax.php' ),
				'nonce'        => wp_create_nonce( 'websamurai_nonce' ),
				'apiUrl'       => rest_url( 'websamurai/v1' ),
				'currentType'  => get_post_type( $post ),
				'currentID'    => $post ? $post->ID : null,
				'slug'         => $post ? $post->post_name : null,
				'locationType' => $location_type,
			)
		);
	}
}
add_action( 'wp_enqueue_scripts', 'websamurai_enqueue_frontend_assets' );
