<?php
/**
 * MCP Server Implementation
 *
 * HTTP-based Model Context Protocol server with WordPress content management tools.
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load MCP tool classes.
require_once WEBSAMURAI_PLUGIN_DIR . 'inc/mcp-tools/class-websamurai-mcp-page-tool.php';
require_once WEBSAMURAI_PLUGIN_DIR . 'inc/mcp-tools/class-websamurai-mcp-post-tool.php';
require_once WEBSAMURAI_PLUGIN_DIR . 'inc/mcp-tools/class-websamurai-mcp-custom-post-tool.php';
require_once WEBSAMURAI_PLUGIN_DIR . 'inc/mcp-tools/class-websamurai-mcp-additional-css-tool.php';
require_once WEBSAMURAI_PLUGIN_DIR . 'inc/mcp-tools/class-websamurai-mcp-theme-info-tool.php';
require_once WEBSAMURAI_PLUGIN_DIR . 'inc/mcp-tools/class-websamurai-mcp-plugin-list-tool.php';
/**
 * MCP Server Class
 */
class WebSamurai_MCP_Server {

	/**
	 * MCP Protocol Version
	 *
	 * @var string
	 */
	const PROTOCOL_VERSION = '2024-11-05';

	/**
	 * Server Information
	 *
	 * @var array
	 */
	private $server_info = array(
		'name'            => 'websamurai-wordpress-server',
		'version'         => '1.0.0',
		'protocolVersion' => self::PROTOCOL_VERSION,
	);

	/**
	 * Registered tools
	 *
	 * @var array
	 */
	private $tools = array();

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->register_tools();
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register available tools
	 *
	 * @return void
	 */
	private function register_tools() {
		// Register built-in echo tool.
		$this->tools['echo'] = array(
			'definition' => array(
				'name'        => 'echo',
				'description' => 'Echoes back the provided message',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'message' => array(
							'type'        => 'string',
							'description' => 'The message to echo back',
						),
					),
					'required'   => array( 'message' ),
				),
			),
			'callback'   => array( $this, 'tool_echo' ),
		);

		// Register WordPress content management tools.
		$this->tools['page_manage'] = array(
			'definition' => WebSamurai_MCP_Page_Tool::get_definition(),
			'callback'   => array( 'WebSamurai_MCP_Page_Tool', 'execute' ),
		);

		$this->tools['post_manage'] = array(
			'definition' => WebSamurai_MCP_Post_Tool::get_definition(),
			'callback'   => array( 'WebSamurai_MCP_Post_Tool', 'execute' ),
		);

		$this->tools['custom_post_manage'] = array(
			'definition' => WebSamurai_MCP_Custom_Post_Tool::get_definition(),
			'callback'   => array( 'WebSamurai_MCP_Custom_Post_Tool', 'execute' ),
		);

		$this->tools['additional_css_manage'] = array(
			'definition' => WebSamurai_MCP_Additional_CSS_Tool::get_definition(),
			'callback'   => array( 'WebSamurai_MCP_Additional_CSS_Tool', 'execute' ),
		);

		$this->tools['theme_info'] = array(
			'definition' => WebSamurai_MCP_Theme_Info_Tool::get_definition(),
			'callback'   => array( 'WebSamurai_MCP_Theme_Info_Tool', 'execute' ),
		);

		$this->tools['plugin_list'] = array(
			'definition' => WebSamurai_MCP_Plugin_List_Tool::get_definition(),
			'callback'   => array( 'WebSamurai_MCP_Plugin_List_Tool', 'execute' ),
		);
	}

	/**
	 * Register REST API routes for MCP server
	 *
	 * @return void
	 */
	public function register_routes() {
		// Main MCP endpoint - handles all MCP requests.
		register_rest_route(
			'websamurai/v1',
			'/mcp',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'handle_mcp_request' ),
				'permission_callback' => function ( $request ) {
					$api_key = $request->get_header( 'X-API-Key' );
					if ( ! $api_key ) {
						// Also check Authorization header.
						$auth_header = $request->get_header( 'Authorization' );
						if ( $auth_header && preg_match( '/Bearer\s+(.+)/i', $auth_header, $matches ) ) {
							$api_key = $matches[1];
						}
					}

					if ( ! WebSamurai_API_Keys::validate_key( $api_key ) ) {
						return false;
					}

					return true;
				},  // This endpoint is API key protected.
			)
		);

		// Server info endpoint.
		register_rest_route(
			'websamurai/v1',
			'/mcp/info',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_server_info' ),
				'permission_callback' => function ( $request ) {
					$api_key = $request->get_header( 'X-API-Key' );
					if ( ! $api_key ) {
						// Also check Authorization header.
						$auth_header = $request->get_header( 'Authorization' );
						if ( $auth_header && preg_match( '/Bearer\s+(.+)/i', $auth_header, $matches ) ) {
							$api_key = $matches[1];
						}
					}

					if ( ! WebSamurai_API_Keys::validate_key( $api_key ) ) {
						return false;
					}

					return true;
				},  // This endpoint is API key protected.
			)
		);
	}

	/**
	 * Handle MCP requests
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error
	 */
	public function handle_mcp_request( $request ) {
		$body = $request->get_json_params();

		if ( empty( $body ) || ! isset( $body['jsonrpc'] ) ) {
			return new WP_Error(
				'invalid_request',
				'Invalid JSON-RPC request',
				array( 'status' => 400 )
			);
		}

		// Extract request details.
		$jsonrpc = $body['jsonrpc'];
		$method  = isset( $body['method'] ) ? $body['method'] : '';
		$params  = isset( $body['params'] ) ? $body['params'] : array();
		$id      = isset( $body['id'] ) ? $body['id'] : null;

		// Validate JSON-RPC version.
		if ( '2.0' !== $jsonrpc ) {
			return $this->create_error_response( $id, -32600, 'Invalid JSON-RPC version' );
		}

		// Route to appropriate handler.
		switch ( $method ) {
			case 'initialize':
				$result = $this->handle_initialize( $params );
				break;

			case 'tools/list':
				$result = $this->handle_tools_list();
				break;

			case 'tools/call':
				$result = $this->handle_tools_call( $params );
				break;

			default:
				return $this->create_error_response( $id, -32601, 'Method not found: ' . $method );
		}

		// Return successful response.
		return rest_ensure_response(
			array(
				'jsonrpc' => '2.0',
				'id'      => $id,
				'result'  => $result,
			)
		);
	}

	/**
	 * Handle initialize request
	 *
	 * @return array
	 */
	private function handle_initialize() {
		return array(
			'protocolVersion' => self::PROTOCOL_VERSION,
			'capabilities'    => array(
				'tools' => array(
					'listChanged' => false,
				),
			),
			'serverInfo'      => $this->server_info,
		);
	}

	/**
	 * Handle tools/list request
	 *
	 * @return array
	 */
	private function handle_tools_list() {
		$tools = array();

		foreach ( $this->tools as $tool ) {
			$tools[] = $tool['definition'];
		}

		return array( 'tools' => $tools );
	}

	/**
	 * Handle tools/call request
	 *
	 * @param array $params Request parameters.
	 * @return array
	 */
	private function handle_tools_call( $params ) {
		$tool_name = isset( $params['name'] ) ? $params['name'] : '';
		$arguments = isset( $params['arguments'] ) ? $params['arguments'] : array();

		// Check if tool exists.
		if ( ! isset( $this->tools[ $tool_name ] ) ) {
			return array(
				'content' => array(
					array(
						'type' => 'text',
						'text' => 'Error: Unknown tool "' . $tool_name . '"',
					),
				),
				'isError' => true,
			);
		}

		// Execute the tool callback.
		$tool     = $this->tools[ $tool_name ];
		$callback = $tool['callback'];

		return call_user_func( $callback, $arguments );
	}

	/**
	 * Echo tool implementation
	 *
	 * @param array $arguments Tool arguments.
	 * @return array
	 */
	private function tool_echo( $arguments ) {
		$message = isset( $arguments['message'] ) ? $arguments['message'] : '';

		if ( empty( $message ) ) {
			return array(
				'content' => array(
					array(
						'type' => 'text',
						'text' => 'Error: message parameter is required',
					),
				),
				'isError' => true,
			);
		}

		return array(
			'content' => array(
				array(
					'type' => 'text',
					'text' => $message,
				),
			),
		);
	}

	/**
	 * Get server info
	 *
	 * @return WP_REST_Response
	 */
	public function get_server_info() {
		$tools_summary = array();

		foreach ( $this->tools as $tool ) {
			$tools_summary[] = array(
				'name'        => $tool['definition']['name'],
				'description' => $tool['definition']['description'],
			);
		}

		return rest_ensure_response(
			array(
				'server'   => $this->server_info,
				'endpoint' => rest_url( 'websamurai/v1/mcp' ),
				'tools'    => $tools_summary,
			)
		);
	}

	/**
	 * Create error response
	 *
	 * @param mixed  $id Request ID.
	 * @param int    $code Error code.
	 * @param string $message Error message.
	 * @return WP_REST_Response
	 */
	private function create_error_response( $id, $code, $message ) {
		return rest_ensure_response(
			array(
				'jsonrpc' => '2.0',
				'id'      => $id,
				'error'   => array(
					'code'    => $code,
					'message' => $message,
				),
			)
		);
	}
}

// Initialize MCP Server.
new WebSamurai_MCP_Server();
