<?php
/**
 * API Key Management
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * API Keys Class
 */
class WebSamurai_API_Keys {

	/**
	 * Option name for storing API keys
	 *
	 * @var string
	 */
	const OPTION_NAME = 'websamurai_api_keys';

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes
	 *
	 * @return void
	 */
	public function register_routes() {
		// Get all API keys.
		register_rest_route(
			'websamurai/v1',
			'/api-keys',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_api_keys' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);

		// Generate new API key.
		register_rest_route(
			'websamurai/v1',
			'/api-keys',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'generate_api_key' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
				'args'                => array(
					'name' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		// Delete API key.
		register_rest_route(
			'websamurai/v1',
			'/api-keys/(?P<id>[a-zA-Z0-9]+)',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_api_key' ),
				'permission_callback' => array( $this, 'check_admin_permission' ),
			)
		);
	}

	/**
	 * Check if user has admin permission
	 *
	 * @return bool
	 */
	public function check_admin_permission() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Get all API keys (without revealing full key)
	 *
	 * @return WP_REST_Response
	 */
	public function get_api_keys() {
		$keys = get_option( self::OPTION_NAME, array() );

		// Mask the keys for display.
		$masked_keys = array();
		foreach ( $keys as $id => $key_data ) {
			$masked_keys[] = array(
				'id'         => $id,
				'name'       => $key_data['name'],
				'key_prefix' => substr( $key_data['key'], 0, 8 ) . '...',
				'created'    => $key_data['created'],
				'last_used'  => isset( $key_data['last_used'] ) ? $key_data['last_used'] : null,
			);
		}

		return rest_ensure_response( $masked_keys );
	}

	/**
	 * Generate new API key
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error
	 */
	public function generate_api_key( $request ) {
		$name = $request->get_param( 'name' );

		// Generate a secure random key.
		$key = 'sk_' . bin2hex( random_bytes( 32 ) );
		$id  = wp_generate_password( 12, false );

		// Get existing keys.
		$keys = get_option( self::OPTION_NAME, array() );

		// Add new key.
		$keys[ $id ] = array(
			'key'     => $key,
			'name'    => $name,
			'created' => current_time( 'mysql' ),
		);

		// Save keys.
		update_option( self::OPTION_NAME, $keys );

		return rest_ensure_response(
			array(
				'id'      => $id,
				'key'     => $key,
				'name'    => $name,
				'created' => $keys[ $id ]['created'],
				'message' => __( 'API key generated successfully. Make sure to copy it now as you won\'t be able to see it again.', 'websamurai' ),
			)
		);
	}

	/**
	 * Delete API key
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response|WP_Error
	 */
	public function delete_api_key( $request ) {
		$id = $request->get_param( 'id' );

		$keys = get_option( self::OPTION_NAME, array() );

		if ( ! isset( $keys[ $id ] ) ) {
			return new WP_Error(
				'key_not_found',
				__( 'API key not found', 'websamurai' ),
				array( 'status' => 404 )
			);
		}

		unset( $keys[ $id ] );
		update_option( self::OPTION_NAME, $keys );

		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'API key deleted successfully', 'websamurai' ),
			)
		);
	}

	/**
	 * Validate API key
	 *
	 * @param string $key API key to validate.
	 * @return bool|string False if invalid, key ID if valid.
	 */
	public static function validate_key( $key ) {
		if ( empty( $key ) ) {
			return false;
		}

		$keys = get_option( self::OPTION_NAME, array() );

		foreach ( $keys as $id => $key_data ) {
			if ( hash_equals( $key_data['key'], $key ) ) {
				// Update last used timestamp.
				$keys[ $id ]['last_used'] = current_time( 'mysql' );
				update_option( self::OPTION_NAME, $keys );

				return $id;
			}
		}

		return false;
	}

	/**
	 * Get API key by name
	 *
	 * @param string $name Name of the API key to find.
	 * @return string|false API key if found, false otherwise.
	 */
	public static function get_key_by_name( $name ) {
		$keys = get_option( self::OPTION_NAME, array() );

		foreach ( $keys as $id => $key_data ) {
			if ( $key_data['name'] === $name ) {
				return $key_data['key'];
			}
		}

		return false;
	}

	/**
	 * Create API key programmatically
	 *
	 * @param string $name Name for the API key.
	 * @return string The generated API key.
	 */
	public static function create_key( $name ) {
		// Generate a secure random key.
		$key = 'sk_' . bin2hex( random_bytes( 32 ) );
		$id  = wp_generate_password( 12, false );

		// Get existing keys.
		$keys = get_option( self::OPTION_NAME, array() );

		// Add new key.
		$keys[ $id ] = array(
			'key'     => $key,
			'name'    => $name,
			'created' => current_time( 'mysql' ),
		);

		// Save keys.
		update_option( self::OPTION_NAME, $keys );

		return $key;
	}

	/**
	 * Get or create the Samurai Remote Key
	 *
	 * This key is used for MCP server communication with the SamuraiWP Control Panel.
	 *
	 * @return string The API key.
	 */
	public static function get_or_create_samurai_remote_key() {
		$key_name = 'Samurai Remote Key';

		// Try to find existing key.
		$existing_key = self::get_key_by_name( $key_name );

		if ( false !== $existing_key ) {
			return $existing_key;
		}

		// Create new key if it doesn't exist.
		$new_key = self::create_key( $key_name );

		return $new_key;
	}
}

// Initialize API Keys.
new WebSamurai_API_Keys();
