<?php
/**
 * AJAX Handler for AI Processing
 *
 * @package WebSamurai
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handle AI processing AJAX request
 *
 * @return void
 */
function websamurai_process_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'websamurai_nonce' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in to use this feature', 'websamurai' ),
			)
		);
	}

	// Check if user is an editor or administrator.
	$user               = wp_get_current_user();
	$is_editor_or_admin = in_array( 'editor', $user->roles, true ) || in_array( 'administrator', $user->roles, true );

	if ( ! $is_editor_or_admin ) {
		wp_send_json_error(
			array(
				'message' => __( 'Access denied. Only editors and administrators can use this AI feature.', 'websamurai' ),
			)
		);
	}

	// Get content from request.
	$content = isset( $_POST['content'] ) ? sanitize_textarea_field( wp_unslash( $_POST['content'] ) ) : '';

	if ( empty( $content ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Content is required', 'websamurai' ),
			)
		);
	}

	// For now, just echo back what was typed.
	// TODO: Add actual AI processing here.
	$response = 'AI Response: ' . $content;

	// Send success response.
	wp_send_json_success(
		array(
			'response' => $response,
			'original' => $content,
		)
	);
}
add_action( 'wp_ajax_websamurai_process', 'websamurai_process_ajax' );
// Note: Removed wp_ajax_nopriv to prevent non-logged-in users from accessing this.

/**
 * Create AIConversation via GraphQL
 *
 * @return void
 */
function websamurai_create_conversation_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'websamurai_nonce' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in', 'websamurai' ),
			)
		);
	}

	// Get auth key.
	$auth_key = get_option( 'websamurai_cp_auth_key' );
	if ( empty( $auth_key ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Not connected to WebSamurai Control Panel', 'websamurai' ),
			)
		);
	}

	// Get parameters from request.
	$subject     = isset( $_POST['subject'] ) ? sanitize_text_field( wp_unslash( $_POST['subject'] ) ) : '';
	$type        = isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : 'WebSamurai';
	$subtype     = isset( $_POST['subtype'] ) ? sanitize_text_field( wp_unslash( $_POST['subtype'] ) ) : '';
	$prompt_text = isset( $_POST['prompt'] ) ? sanitize_textarea_field( wp_unslash( $_POST['prompt'] ) ) : '';

	// Create the prompt array structure with the sanitized text.
	$prompt = array(
		'prompts' => array(
			array(
				'role'    => 'user',
				'content' => $prompt_text,
			),
		),
	);

	if ( isset( $_POST['context'] ) ) {
		$context                 = array();
		$postcontext             = json_decode( wp_unslash( $_POST['context'] ), true ); // this is sanitized below.
		$context['currentType']  = sanitize_text_field( $postcontext['currentType'] );
		$context['currentID']    = sanitize_text_field( $postcontext['currentID'] );
		$context['slug']         = sanitize_text_field( $postcontext['slug'] );
		$context['locationType'] = sanitize_text_field( $postcontext['locationType'] );
		$prompt['context']       = $context;
	} else {
		$prompt['context'] = array();
	}
	$prompt['context']['SamuraiSite'] = get_option( 'websamurai_cp_site_id' );
	$prompt['context']['WPSiteURL']   = get_site_url();

	// Determine GraphQL endpoint URL.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	// Build GraphQL mutation.
	$graphql_query = array(
		'query'     => 'mutation ($Subject: String, $Type: String, $SubType: String, $Prompt: CustomObject, $DoneResponse: Boolean) {
			CreateAIConversation (Subject: $Subject, Type: $Type, SubType: $SubType, Prompt: $Prompt, DoneResponse: $DoneResponse) {
				id
				Subject
				Type
				SubType
				Prompt
				DoneResponse
			}
		}',
		'variables' => array(
			'Subject'      => $subject,
			'Type'         => $type,
			'SubType'      => $subtype,
			'Prompt'       => $prompt,
			'DoneResponse' => false,
		),
	);

	// Make GraphQL request.
	$response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $graphql_query ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to connect to server: %s', 'websamurai' ), $response->get_error_message() ),
			)
		);
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( 200 !== $status_code ) {
		wp_send_json_error(
			array(
				/* translators: 1: Status code. */
				'message' => sprintf( __( 'Server returned error: %s', 'websamurai' ), $status_code ),
				'data'    => $data,
			)
		);
	}

	// Check for GraphQL errors.
	if ( isset( $data['errors'] ) && ! empty( $data['errors'] ) ) {
		$errs = array_map( function( $error ) {
			return isset( $error['message'] ) ? $error['message'] : 'Unknown error';
		}, $data['errors'] );
		wp_send_json_error(
			array(
				'message' => __( 'Error: ' . implode( '\n', $errs ), 'websamurai' ),
				'errors'  => $data['errors'],
			)
		);
	}

	// Extract conversation data.
	$conversation = isset( $data['data']['CreateAIConversation'] ) ? $data['data']['CreateAIConversation'] : null;

	if ( ! $conversation ) {
		wp_send_json_error(
			array(
				'message' => __( 'No conversation data received from server', 'websamurai' ),
			)
		);
	}

	wp_send_json_success(
		array(
			'conversation' => $conversation,
		)
	);
}
add_action( 'wp_ajax_websamurai_create_conversation', 'websamurai_create_conversation_ajax' );

/**
 * Get AIConversation via GraphQL
 *
 * @return void
 */
function websamurai_get_conversation_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'websamurai_nonce' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in', 'websamurai' ),
			)
		);
	}

	// Get auth key.
	$auth_key = get_option( 'websamurai_cp_auth_key' );
	if ( empty( $auth_key ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Not connected to WebSamurai Control Panel', 'websamurai' ),
			)
		);
	}

	// Get conversation ID.
	$conversation_id = isset( $_POST['conversation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['conversation_id'] ) ) : '';

	if ( empty( $conversation_id ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Conversation ID is required', 'websamurai' ),
			)
		);
	}

	// Determine GraphQL endpoint URL.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	// Build GraphQL query.
	$graphql_query = array(
		'query'     => 'query ($id: ID!) {
			AIConversation(id: $id) {
				id
				Subject
				Type
				SubType
				Prompt
				DoneResponse
				UpdatedDate
			}
		}',
		'variables' => array(
			'id' => $conversation_id,
		),
	);

	// Make GraphQL request.
	$response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $graphql_query ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to connect to server: %s', 'websamurai' ), $response->get_error_message() ),
			)
		);
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( 200 !== $status_code ) {
		wp_send_json_error(
			array(
				/* translators: 1: Status code. */
				'message' => sprintf( __( 'Server returned error: %s', 'websamurai' ), $status_code ),
				'data'    => $data,
			)
		);
	}

	// Check for GraphQL errors.
	if ( isset( $data['errors'] ) && ! empty( $data['errors'] ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'GraphQL query error', 'websamurai' ),
				'errors'  => $data['errors'],
			)
		);
	}

	// Extract conversation data.
	$conversation = isset( $data['data']['AIConversation'] ) ? $data['data']['AIConversation'] : null;

	if ( ! $conversation ) {
		wp_send_json_error(
			array(
				'message' => __( 'No conversation data received from server', 'websamurai' ),
			)
		);
	}

	wp_send_json_success(
		array(
			'conversation' => $conversation,
		)
	);
}
add_action( 'wp_ajax_websamurai_get_conversation', 'websamurai_get_conversation_ajax' );

/**
 * Update AIConversation via GraphQL
 *
 * @return void
 */
function websamurai_update_conversation_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'websamurai_nonce' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in', 'websamurai' ),
			)
		);
	}

	// Get auth key.
	$auth_key = get_option( 'websamurai_cp_auth_key' );
	if ( empty( $auth_key ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Not connected to WebSamurai Control Panel', 'websamurai' ),
			)
		);
	}

	// Get parameters from request.
	$conversation_id = isset( $_POST['conversation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['conversation_id'] ) ) : '';
	$new_message     = isset( $_POST['message'] ) ? sanitize_textarea_field( wp_unslash( $_POST['message'] ) ) : '';

	if ( empty( $conversation_id ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Conversation ID is required', 'websamurai' ),
			)
		);
	}

	if ( empty( $new_message ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Message is required', 'websamurai' ),
			)
		);
	}

	// First, get the existing conversation to retrieve current Prompt.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	$get_query = array(
		'query'     => 'query ($id: ID!) {
			AIConversation(id: $id) {
				id
				Prompt
			}
		}',
		'variables' => array(
			'id' => $conversation_id,
		),
	);

	$get_response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $get_query ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $get_response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to get conversation: %s', 'websamurai' ), $get_response->get_error_message() ),
			)
		);
	}

	$get_status = wp_remote_retrieve_response_code( $get_response );
	$get_body   = wp_remote_retrieve_body( $get_response );
	$get_data   = json_decode( $get_body, true );

	
	if ( 200 !== $get_status || isset( $get_data['data']['errors'] ) ) {
		if( isset( $get_data['data']['errors'] ) ) {
			$errors = $get_data['data']['errors'];
		} else {
			$errors = array();
		}
		wp_send_json_error(
			array(
				'message' => __( 'Failed to retrieve existing conversation: ' . implode( '\n', $errors ), 'websamurai' ),
			)
		);
	}

	$existing_conversation = isset( $get_data['data']['AIConversation'] ) ? $get_data['data']['AIConversation'] : null;

	if ( ! $existing_conversation ) {
		wp_send_json_error(
			array(
				'message' => __( 'Conversation not found', 'websamurai' ),
			)
		);
	}

	// Get existing Prompt and add new user message.
	$prompt = $existing_conversation['Prompt'];
	if ( ! isset( $prompt['prompts'] ) || ! is_array( $prompt['prompts'] ) ) {
		$prompt['prompts'] = array();
	}

	// Add the new user message to the prompts array.
	$prompt['prompts'][] = array(
		'role'    => 'user',
		'content' => $new_message,
	);

	// Build GraphQL mutation to update the conversation.
	$update_mutation = array(
		'query'     => 'mutation ($id: ID!, $Prompt: CustomObject, $DoneResponse: Boolean) {
			UpdateAIConversation (id: $id, Prompt: $Prompt, DoneResponse: $DoneResponse) {
				id
				Subject
				Type
				SubType
				Prompt
				DoneResponse
			}
		}',
		'variables' => array(
			'id'     => $conversation_id,
			'Prompt' => $prompt,
		),
	);

	// Make GraphQL request to update.
	$update_response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $update_mutation ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $update_response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to update conversation: %s', 'websamurai' ), $update_response->get_error_message() ),
			)
		);
	}

	$update_status = wp_remote_retrieve_response_code( $update_response );
	$update_body   = wp_remote_retrieve_body( $update_response );
	$update_data   = json_decode( $update_body, true );

	if ( 200 !== $update_status ) {
		wp_send_json_error(
			array(
				/* translators: 1: Status code. */
				'message' => sprintf( __( 'Server returned error: %s', 'websamurai' ), $update_status ),
				'data'    => $update_data,
			)
		);
	}

	// Check for GraphQL errors.
	if ( isset( $update_data['errors'] ) && ! empty( $update_data['errors'] ) ) {
		$errs = array_map( function( $error ) {
			return isset( $error['message'] ) ? $error['message'] : 'Unknown error';
		}, $update_data['errors'] );
		wp_send_json_error(
			array(
				'message' => __( 'Error: ' . implode( '\n', $errs ), 'websamurai' ),
				'errors'  => $update_data['errors'],
			)
		);
	}

	// Extract updated conversation data.
	$conversation = isset( $update_data['data']['UpdateAIConversation'] ) ? $update_data['data']['UpdateAIConversation'] : null;

	if ( ! $conversation ) {
		wp_send_json_error(
			array(
				'message' => __( 'No conversation data received from server', 'websamurai' ),
			)
		);
	}

	wp_send_json_success(
		array(
			'conversation' => $conversation,
		)
	);
}
add_action( 'wp_ajax_websamurai_update_conversation', 'websamurai_update_conversation_ajax' );

/**
 * Generate AI Image via GraphQL
 *
 * @return void
 */
function websamurai_generate_image_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'wp_rest' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in', 'websamurai' ),
			)
		);
	}

	// Get auth key.
	$auth_key = get_option( 'websamurai_cp_auth_key' );
	if ( empty( $auth_key ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Not connected to WebSamurai Control Panel', 'websamurai' ),
			)
		);
	}

	// Get image description from request.
	$description = isset( $_POST['description'] ) ? sanitize_textarea_field( wp_unslash( $_POST['description'] ) ) : '';

	if ( empty( $description ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Image description is required', 'websamurai' ),
			)
		);
	}

	// Determine GraphQL endpoint URL.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	// Build prompt for image generation.
	$prompt = array(
		'context' => array(
			'SamuraiSite' => get_option( 'websamurai_cp_site_id' ),
			'WPSiteURL'   => get_site_url(),
		),
		'prompts' => array(
			array(
				'text' => $description,
			),
		),
	);

	// Build GraphQL mutation.
	$graphql_query = array(
		'query'     => 'mutation ($Subject: String, $Type: String, $SubType: String, $Prompt: CustomObject, $DoneResponse: Boolean) {
			CreateAIConversation (Subject: $Subject, Type: $Type, SubType: $SubType, Prompt: $Prompt, DoneResponse: $DoneResponse) {
				id
				Subject
				Type
				SubType
				Prompt
				DoneResponse
			}
		}',
		'variables' => array(
			'Subject'      => 'AI Image Generation',
			'Type'         => 'WebSamurai',
			'SubType'      => 'image',
			'Prompt'       => $prompt,
			'DoneResponse' => false,
		),
	);

	// Make GraphQL request.
	$response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $graphql_query ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to connect to server: %s', 'websamurai' ), $response->get_error_message() ),
			)
		);
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( 200 !== $status_code ) {
		wp_send_json_error(
			array(
				/* translators: 1: Status code. */
				'message' => sprintf( __( 'Server returned error: %s', 'websamurai' ), $status_code ),
				'data'    => $data,
			)
		);
	}

	// Check for GraphQL errors.
	if ( isset( $data['errors'] ) && ! empty( $data['errors'] ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'GraphQL query error', 'websamurai' ),
				'errors'  => $data['errors'],
			)
		);
	}

	// Extract conversation data.
	$conversation = isset( $data['data']['CreateAIConversation'] ) ? $data['data']['CreateAIConversation'] : null;

	if ( ! $conversation ) {
		wp_send_json_error(
			array(
				'message' => __( 'No conversation data received from server', 'websamurai' ),
			)
		);
	}

	wp_send_json_success(
		array(
			'conversation' => $conversation,
		)
	);
}
add_action( 'wp_ajax_websamurai_generate_image', 'websamurai_generate_image_ajax' );

/**
 * Poll AI Image Status via GraphQL
 *
 * @return void
 */
function websamurai_poll_image_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'wp_rest' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in', 'websamurai' ),
			)
		);
	}

	// Get auth key.
	$auth_key = get_option( 'websamurai_cp_auth_key' );
	if ( empty( $auth_key ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Not connected to WebSamurai Control Panel', 'websamurai' ),
			)
		);
	}

	// Get conversation ID.
	$conversation_id = isset( $_POST['conversation_id'] ) ? sanitize_text_field( wp_unslash( $_POST['conversation_id'] ) ) : '';

	if ( empty( $conversation_id ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Conversation ID is required', 'websamurai' ),
			)
		);
	}

	// Determine GraphQL endpoint URL.
	$graphql_url = defined( 'LOCALDEV' ) && LOCALDEV ? 'http://uchideshi:4500/graphql' : 'https://cp.websamurai.io/graphql';

	// Build GraphQL query.
	$graphql_query = array(
		'query'     => 'query ($id: ID!) {
			AIConversation(id: $id) {
				id
				Subject
				Type
				SubType
				Prompt
				DoneResponse
			}
		}',
		'variables' => array(
			'id' => $conversation_id,
		),
	);

	// Make GraphQL request.
	$response = wp_remote_post(
		$graphql_url,
		array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'x-auth-code'  => $auth_key,
			),
			'body'    => wp_json_encode( $graphql_query ),
			'timeout' => 15,
		)
	);

	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to connect to server: %s', 'websamurai' ), $response->get_error_message() ),
			)
		);
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	$body        = wp_remote_retrieve_body( $response );
	$data        = json_decode( $body, true );

	if ( 200 !== $status_code ) {
		wp_send_json_error(
			array(
				/* translators: 1: Status code. */
				'message' => sprintf( __( 'Server returned error: %s', 'websamurai' ), $status_code ),
				'data'    => $data,
			)
		);
	}

	// Check for GraphQL errors.
	if ( isset( $data['errors'] ) && ! empty( $data['errors'] ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'GraphQL query error', 'websamurai' ),
				'errors'  => $data['errors'],
			)
		);
	}

	// Extract conversation data.
	$conversation = isset( $data['data']['AIConversation'] ) ? $data['data']['AIConversation'] : null;

	if ( ! $conversation ) {
		wp_send_json_error(
			array(
				'message' => __( 'No conversation data received from server', 'websamurai' ),
			)
		);
	}

	wp_send_json_success(
		array(
			'conversation' => $conversation,
		)
	);
}
add_action( 'wp_ajax_websamurai_poll_image', 'websamurai_poll_image_ajax' );

/**
 * Add generated image to media library
 *
 * @return void
 */
function websamurai_add_image_to_media_ajax() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'wp_rest' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Invalid nonce', 'websamurai' ),
			)
		);
	}

	// Check if user is logged in.
	if ( ! is_user_logged_in() ) {
		wp_send_json_error(
			array(
				'message' => __( 'You must be logged in', 'websamurai' ),
			)
		);
	}

	// Check user capabilities.
	if ( ! current_user_can( 'upload_files' ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'You do not have permission to upload files', 'websamurai' ),
			)
		);
	}

	// Get image URL and description.
	$image_url   = isset( $_POST['image_url'] ) ? esc_url_raw( wp_unslash( $_POST['image_url'] ) ) : '';
	$description = isset( $_POST['description'] ) ? sanitize_text_field( wp_unslash( $_POST['description'] ) ) : '';

	if ( empty( $image_url ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Image URL is required', 'websamurai' ),
			)
		);
	}

	// Download the image using wp_remote_get (works with http localhost URLs).
	require_once ABSPATH . 'wp-admin/includes/media.php';
	require_once ABSPATH . 'wp-admin/includes/file.php';
	require_once ABSPATH . 'wp-admin/includes/image.php';

	// Use wp_remote_get to download the file.
	$response = wp_remote_get(
		$image_url,
		array(
			'timeout'     => 10,
			'redirection' => 5,
			'sslverify'   => false, // Allow self-signed certs for local dev.
		)
	);

	if ( is_wp_error( $response ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to download image: %1$s from URL: %2$s', 'websamurai' ), $response->get_error_message(), $image_url ),
			)
		);
	}

	$http_code = wp_remote_retrieve_response_code( $response );

	if ( 200 !== $http_code ) {
		wp_send_json_error(
			array(
				/* translators: 1: Image URL, 2: HTTP code. */
				'message' => sprintf( __( 'Failed to download image from URL: %1$s (HTTP %2$s)', 'websamurai' ), $image_url, $http_code ),
			)
		);
	}

	// Get the image data.
	$image_data = wp_remote_retrieve_body( $response );

	if ( empty( $image_data ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Downloaded image is empty', 'websamurai' ),
			)
		);
	}

	// Create a temporary file and write the image data.
	$tmp_file = wp_tempnam( $image_url );

	if ( ! $tmp_file ) {
		wp_send_json_error(
			array(
				'message' => __( 'Failed to create temporary file', 'websamurai' ),
			)
		);
	}

	// Initialize WP_Filesystem.
	global $wp_filesystem;
	if ( ! WP_Filesystem() ) {
		wp_delete_file( $tmp_file );
		wp_send_json_error(
			array(
				'message' => __( 'Failed to initialize filesystem', 'websamurai' ),
			)
		);
	}

	$writeret = $wp_filesystem->put_contents( $tmp_file, $image_data );

	if ( false === $writeret ) {
		wp_delete_file( $tmp_file );
		wp_send_json_error(
			array(
				'message' => __( 'Failed to write image data to temporary file', 'websamurai' ),
			)
		);
	}

	// Prepare file array for wp_handle_sideload.
	$file_array = array(
		'name'     => 'ai-generated-' . time() . '.png',
		'tmp_name' => $tmp_file,
	);

	// Upload file to media library.
	$attachment_id = media_handle_sideload( $file_array, 0, $description );

	// Delete temp file.
	if ( file_exists( $tmp_file ) ) {
		wp_delete_file( $tmp_file );
	}

	if ( is_wp_error( $attachment_id ) ) {
		wp_send_json_error(
			array(
				/* translators: 1: Error message. */
				'message' => sprintf( __( 'Failed to add image to media library: %s', 'websamurai' ), $attachment_id->get_error_message() ),
			)
		);
	}

	// Set alt text.
	update_post_meta( $attachment_id, '_wp_attachment_image_alt', $description );

	// Get attachment details.
	$attachment     = get_post( $attachment_id );
	$attachment_url = wp_get_attachment_url( $attachment_id );

	wp_send_json_success(
		array(
			'attachment_id'  => $attachment_id,
			'attachment_url' => $attachment_url,
			'attachment'     => $attachment,
		)
	);
}
add_action( 'wp_ajax_websamurai_add_image_to_media', 'websamurai_add_image_to_media_ajax' );
